﻿using System;
using System.Xml;
using System.Net;
using System.Text;
using System.Net.Sockets;
using System.Collections.Generic;
using UnityEngine;

namespace Movella.Xsens
{
    public class XsensRemoteControl
    {
        private readonly UdpClient _udpClient = new();

        private readonly Dictionary<string, string> _commandMessages = new()
        {
            { "first", "<NavigateToStartReq/>" },
            { "previous", "<PreviousFrameReq/>" },
            { "togglePlayback", "<PlayPauseReq/>" },
            { "next", "<NextFrameReq/>" },
            { "last", "<NavigateToEndReq/>" },
            { "toggleLoop", "<ToggleRepeatReq/>" },
            { "jumpToFrame", "<JumpToFrameReq/>" },
            { "moveToOrigin", "<MoveCharacterToOriginReq/>" },
            { "resetAxis", "<ResetAxisReq/>" },
            { "status", "<SessionStatusReq>" },
            { "info", "<SessionInfoReq/>" }
        };

        private readonly object _lock = new object();

        public string SendMessage(string request, string ipAddress, int port, bool getResponse = false)
        {
            if (!_commandMessages.TryGetValue(request, out string message))
            {
                if (
                    request.StartsWith("<CaptureName>") || 
                    request.StartsWith("<CaptureStart>") || 
                    request.StartsWith("<CaptureStop>") ||
                    request.StartsWith("<JumpToFrameReq")
                    )
                {
                    message = request;
                }
                else
                {
                    Debug.LogError($"Request '{request}' not found in command messages.");
                    return null;
                }
            }

            lock (_lock)
            {
                try // Clear any pending messages in the reception buffer
                {
                    _udpClient.Client.ReceiveTimeout = 1; // ms
                    IPEndPoint remoteEndPoint = new IPEndPoint(IPAddress.Any, port);
                    while (_udpClient.Available > 0)
                    {
                        _udpClient.Receive(ref remoteEndPoint);
                    }

                    byte[] data = Encoding.UTF8.GetBytes(message);
                    _udpClient.Send(data, data.Length, ipAddress, port);

                    if (getResponse)
                    {
                        _udpClient.Client.ReceiveTimeout = 250; //  250 == .25 seconds
                        byte[] receivedData = _udpClient.Receive(ref remoteEndPoint);
                        string response = Encoding.UTF8.GetString(receivedData);

                        return response;
                    }
                }
                catch (SocketException)
                {
                    return null;
                    // Debug.LogWarning("Error sending or receiving UDP message. Check remote control port\n" + e);
                }
            }

            return null;
        }

        public SessionStatusAck ParseStatusMessage(string statusMessage)
        {
            var doc = new XmlDocument();
            doc.LoadXml(statusMessage);

            if (doc.DocumentElement == null) return null;
            var status = new SessionStatusAck
            {
                CurrentFrame =
                    doc.DocumentElement != null && int.TryParse(doc.DocumentElement.GetAttribute("currentFrame"),
                        out int currentFrame)
                        ? currentFrame
                        : 0,
                // CurrentFrameGlobalTime = doc.DocumentElement != null && long.TryParse(doc.DocumentElement.GetAttribute("currentFrameGlobalTime"), out long currentFrameGlobalTime) ? currentFrameGlobalTime : 0L,
                // CurrentFrameTime = doc.DocumentElement != null && int.TryParse(doc.DocumentElement.GetAttribute("currentFrameTime"), out int currentFrameTime) ? currentFrameTime : 0,
                // IsCalibrating = doc.DocumentElement != null && doc.DocumentElement.GetAttribute("isCalibrating").Equals("TRUE", StringComparison.OrdinalIgnoreCase),
                // IsFlushing = doc.DocumentElement != null && doc.DocumentElement.GetAttribute("isFlushing").Equals("TRUE", StringComparison.OrdinalIgnoreCase),
                // IsPaused = doc.DocumentElement != null && doc.DocumentElement.GetAttribute("isPaused").Equals("TRUE", StringComparison.OrdinalIgnoreCase),
                IsPlaying = doc.DocumentElement != null && doc.DocumentElement.GetAttribute("isPlaying")
                    .Equals("TRUE", StringComparison.OrdinalIgnoreCase),
                IsRecording = doc.DocumentElement != null && doc.DocumentElement.GetAttribute("isRecording")
                    .Equals("TRUE", StringComparison.OrdinalIgnoreCase),
                IsRepeating = doc.DocumentElement != null && doc.DocumentElement.GetAttribute("isRepeating")
                    .Equals("TRUE", StringComparison.OrdinalIgnoreCase),
                // PlaySpeedRatio = doc.DocumentElement != null && float.TryParse(doc.DocumentElement.GetAttribute("playSpeedRatio"), out float playSpeedRatio) ? playSpeedRatio : 0f
            };

            return status;
        }

        public SessionInfoAck ParseInfoMessage(string infoMessage)
        {
            var doc = new XmlDocument();
            doc.LoadXml(infoMessage);

            if (doc.DocumentElement == null) return null;
            var info = new SessionInfoAck
            {
                DurationFrames =
                    doc.DocumentElement != null && int.TryParse(doc.DocumentElement.GetAttribute("fileDurationFrame"),
                        out int fileDurationFrame)
                        ? fileDurationFrame
                        : 0,
                // DurationMs = doc.DocumentElement != null && int.TryParse(doc.DocumentElement.GetAttribute("fileDurationMs"), out int fileDurationMs) ? fileDurationMs : 0, 
                // IsLive = doc.DocumentElement != null && doc.DocumentElement.GetAttribute("isLive").Equals("TRUE", StringComparison.OrdinalIgnoreCase),
                // TakeNumber = doc.DocumentElement != null && int.TryParse(doc.DocumentElement.GetAttribute("recordingTrial"), out int recordingTrial) ? recordingTrial : 0,
                // SessionPath = doc.DocumentElement.GetAttribute("sessionPath"),
            };

            return info;
        }
    }
}

public class SessionStatusAck
{
    public int CurrentFrame { get; set; }

    // public long CurrentFrameGlobalTime { get; set; }
    // public int CurrentFrameTime { get; set; }
    // public bool IsCalibrating { get; set; }
    // public bool IsFlushing { get; set; }
    // public bool IsPaused { get; set; }
    public bool IsPlaying { get; set; }
    public bool IsRecording { get; set; }

    public bool IsRepeating { get; set; }
    // public float PlaySpeedRatio { get; set; }
}

public class SessionInfoAck
{
    public int DurationFrames { get; set; }

    // public int DurationMs { get; set; }
    // public bool IsLive { get; set; }
    // public int TakeNumber { get; set; }
    public string TakeName { get; set; }
    // public string SessionPath { get; set; }
}