using System;
using System.Net;
using System.Net.Sockets;
using System.Text;
using System.Threading;
using System.Collections.Generic;
using System.Linq;
using UnityEngine;
using Unity.LiveCapture;

namespace Movella.Xsens
{
    /// <summary>
    /// Creates a new Thread that listens for incoming data on a specified port. Parses the data packet and converts
    /// it to an XsMvnPose object. The frame is then added to m_frames[] at the characterID index.
    /// Metadata and Scale streams are parsed and the results are added to the frame object 
    /// The XsensClient class is responsible for managing the connection to n Xsens device, receiving data packets, and
    /// converting them into XsMvnPose objects. It maintains an array of FrameData objects and corresponding mutexes for
    /// thread-safe access.
    /// Xsens Network Streamer Protocol:
    ///     https://www.xsens.com/hubfs/Downloads/Manuals/MVN_real-time_network_streaming_protocol_specification.pdf
    /// </summary>
    class XsensClient
    {
        public bool IsStarted => m_Thread is { IsAlive: true };

        public FrameRate FrameRate { get; set; } = StandardFrameRate.FPS_24_00;

        public bool IsConnected { get; private set; }
        public int Port { get; private set; }

        public event Func<int, FrameData, bool> FrameDataReceivedAsync;
        public event Action Disconnected;

        enum StreamingProtocol
        {
            SPPoseEuler = 1,
            SPPoseQuaternion = 2,
            SPPosePositions = 3,
            SPTagPositionsLegacy = 4,
            SPPoseUnity3D = 5,
            SPMetaScalingLegacy = 10,
            SPMetaPropInfoLegacy = 11,
            SPMetaCharacter = 12,
            SPMetaScale = 13,
            SPTimecode = 25,
        };

        Thread m_Thread;
        UdpClient m_Client;
        (Timecode tc, FrameRate rate) m_Timecode;

        public List<EntityMetadata> MetaCollection { get; } = new List<EntityMetadata>();
        private int _prevMetaMsgID;
        private int _metaEntityCount = 0;
        
        public List<EntityScaleMetadata> ScaleMetaCollection { get; } = new List<EntityScaleMetadata>();
        private int _prevScaleMsgID;
        private int _scaleEntityCount = 0;
        
        public List<bool> StatusCollection { get; } = Enumerable.Repeat(true, XsensConstants.NumStreams).ToList();
        
        public FrameData[] m_Frames = new FrameData[XsensConstants.NumStreams];
        object[] m_FrameMutexes = new object[XsensConstants.NumStreams];

        public XsensClient()
        {
            for (int i = 0; i < XsensConstants.NumStreams; i++)
                m_FrameMutexes[i] = new object();
        }

        public void Connect(int port)
        {
            if (port <= 0 || port > 0xFFFF)
            {
                Debug.LogError($"{nameof(XsensClient)}: Tried to connect with an invalid port: {port}");
                return;
            }

            if (!IsConnected)
            {
                try
                {
                    // Start a new thread to listen for incoming data Via ListenForDataAsync
                    m_Thread = new Thread(() => ListenForDataAsync(port));
                    m_Thread.IsBackground = true;
                    m_Thread.Start();
                }
                catch (Exception e)
                {
                    m_Thread = null;
                    IsConnected = false;

                    Debug.Log($"{nameof(XsensClient)}({port}): Thread start exception {e}");
                }
            }
        }

        public void Disconnect()
        {
            IsConnected = false;

            if (m_Thread != null)
            {
                if (!m_Thread.Join(2000))
                    m_Thread.Abort();

                m_Thread = null;
            }

            m_Client?.Close();
            m_Client = null;

            Disconnected?.Invoke();
        }

        public FrameData GetFrame(int characterID)
        {
            if (IsConnected &&
                characterID >= 0 &&
                characterID < XsensConstants.NumStreams)
            {
                lock (m_FrameMutexes[characterID])
                    return m_Frames[characterID];
            }

            return default;
        }

        void ListenForDataAsync(int port)
        {
            try
            {
                Port = port;

                IPEndPoint endPoint = new IPEndPoint(IPAddress.Any, port);

                m_Client = new UdpClient(port);

                IsConnected = true;

                while (IsConnected)
                {
                    var receiveBytes = m_Client.Receive(ref endPoint);

                    string result = "";
                    result += (char)receiveBytes[4];
                    result += (char)receiveBytes[5];
                    StreamingProtocol packId = (StreamingProtocol)int.Parse(result);
                    int msgEntityID = receiveBytes[16];

                    switch (packId)
                    {
                        // Update the packet's corresponding m_Frame and update its metadata based on data received from
                        // other message types that have been received since the last quaternion message.
                        case StreamingProtocol.SPPoseQuaternion:
                        {
                            if (
                                msgEntityID < 0 || 
                                receiveBytes.Length <= 15 || 
                                msgEntityID >= XsensConstants.NumStreams
                                )
                                break;
                            
                            var frame = ParseQuaternionPacket(receiveBytes); // Parse to frame
                            frame = UpdateFrameMetadata(frame, msgEntityID);
                            
                            if (FrameDataReceivedAsync?.Invoke(msgEntityID, frame) ?? false)
                                break;
                            
                            // Store the FrameData object in the m_Frames array at the characterID index
                            lock (m_FrameMutexes[msgEntityID])
                                m_Frames[msgEntityID] = frame;
                            break;
                        }

                        case StreamingProtocol.SPMetaCharacter:
                        {
                            if (
                                msgEntityID < 0 || 
                                receiveBytes.Length <= XsensConstants.HeaderLength || 
                                msgEntityID >= XsensConstants.NumStreams
                                )
                                break;
                            
                            EntityMetadata metadata = ParseMetadataPacket(receiveBytes);
                            
                            if (CheckEntityCountChanged(msgEntityID))
                            {
                                MetaCollection.Clear();
                                MetaCollection.Add(metadata);
                                _metaEntityCount = _prevMetaMsgID + 1;
                            }
                            else
                            {
                                if (MetaCollection.Count < msgEntityID + 1)
                                    MetaCollection.Add(metadata);
                                else
                                    MetaCollection[msgEntityID] = metadata;
                            }
                            
                            _prevMetaMsgID = msgEntityID;
                            break;
                        }

                        case StreamingProtocol.SPMetaScale:
                            // Scale Message is used only to determine if a stream represents an actor or object entity
                        {
                            if (
                                msgEntityID < 0 || 
                                msgEntityID >= XsensConstants.NumStreams ||
                                receiveBytes.Length <= XsensConstants.HeaderLength ||
                                !ValidateScalePacket(receiveBytes)
                                )
                                break;
                                                        
                            var streamScale = msgEntityID < ScaleMetaCollection.Count 
                                ? ScaleMetaCollection[msgEntityID] 
                                : new EntityScaleMetadata();

                            streamScale = ParseScalePacket(receiveBytes, streamScale); 
                            
                            if (CheckEntityCountChanged(msgEntityID))
                            {
                                _scaleEntityCount = _prevScaleMsgID + 1;
                                ScaleMetaCollection.Clear(); // remove old entities
                                ScaleMetaCollection.Add(streamScale);
                            }
                            else
                            {
                                int entityCount = msgEntityID + 1;
                                if (ScaleMetaCollection.Count < entityCount)
                                    ScaleMetaCollection.Add(streamScale);
                                else
                                    ScaleMetaCollection[msgEntityID] = streamScale;
                                _prevScaleMsgID = msgEntityID;
                            }
                            break;
                        }

                        case StreamingProtocol.SPTimecode:
                        {
                            // 12 byte string formatted as such HH:MM:SS.mmm
                            // MVN strings are UTF-8 encoded
                            if (receiveBytes.Length > 35)
                            {
                                var timecode = Encoding.UTF8.GetString(receiveBytes[28..39]);

                                if (DateTime.TryParse(timecode, out var timestamp))
                                {
                                    var totalSeconds = timestamp.Hour * 3600 + timestamp.Minute * 60 + timestamp.Second + (float)timestamp.Millisecond / 1000;
                                    m_Timecode.tc = Timecode.FromSeconds(FrameRate, totalSeconds);
                                    m_Timecode.rate = FrameRate;
                                }
                            }
                            break;
                        }
                    }
                }
            }
            catch (SocketException socketException)
            {
                Debug.LogError($"{nameof(XsensClient)}({port}): " + socketException.Message);
            }
            catch (System.IO.IOException ioException)
            {
                if (IsConnected) // if not connected, then ignore this exception because it's a normal stream.Read interruption caused by closing the socket in Disconnect().
                    Debug.LogError($"{nameof(XsensClient)}({port}): " + ioException.Message);
            }
            finally
            {
                m_Client?.Close();
                m_Client = null;
                IsConnected = false;
            }
        }

        FrameData ParseQuaternionPacket(byte[] data)
        {
            XsDataPacket dataPacket = new XsQuaternionPacket(data);
            // This is where the data is parsed into a FrameData object. An XsMVNPose is created from the data packet
            // after using XsDataPacket.getPose() the positions and orientations are stored in the FrameData object
            XsMvnPose pose = dataPacket.getPose();

            if (pose != null)
            {
                var frame = new FrameData()
                {
                    TC = m_Timecode.tc,
                    FrameRate = m_Timecode.rate,
                    SegmentCount = data.Length / 32,
                    Positions = pose.positions,
                    Orientations = pose.orientations,
                    NumProps = pose.MvnCurrentPropCount,
                    entityScaleData = new EntityScaleMetadata(),
                    metadata = new EntityMetadata(),
                    enabled = true,
                };

                return frame;
            }

            return default;
        }

        EntityMetadata ParseMetadataPacket(byte[] data)
        {
            var metaString = Encoding.UTF8.GetString(data[28..]);
            var items = metaString.Split("\n");
            return new EntityMetadata()
            {
                entityColor = items[0].Split(":")[1],
                totalEntityCount = int.Parse(items[1].Split(":")[1]),
                entityName = items[2].Split(":")[1],
                timeOffset = int.Parse(items[3].Split(":")[1]),
            };
        }

        EntityScaleMetadata ParseScalePacket(byte[] data, EntityScaleMetadata entityScale)
        {
            if (XsensConstants.HeaderLength +4 > data.Length)
                return entityScale;
            
            int currentByteIndex = XsensConstants.HeaderLength;
            int segmentCount = BitConverter.ToInt32(data, currentByteIndex);
            currentByteIndex += 4;
            var segments = new List<string>();
            
            for (int i = 0; i < segmentCount; i++)
            {
                byte nameLength = data[currentByteIndex];
                string segmentName = Encoding.UTF8.GetString(data, currentByteIndex + 1, nameLength);
                segments.Add(segmentName);
                currentByteIndex += (4 + nameLength);
                currentByteIndex += 12; // Skip the segment coordinates. Body Dimensions are not used
            }
            
            return new EntityScaleMetadata()
            {
                isObjects = GetIsObject(segments),
                segmentNames = segments,
                propSegmentNames = new List<string>(),
                fingerSegmentNames = new List<string>(),
            };
        }
        
        bool GetIsObject(List<string> segments)
        {
            bool hasPelvis = segments.Contains("Pelvis");
            bool hasUpperLeg = segments.Contains("RightUpperLeg") ||
                               segments.Contains("LeftUpperLeg");
            return !hasPelvis && !hasUpperLeg;
        }

        bool ValidateScalePacket(byte[] data)
        {
            if (XsensConstants.HeaderLength + 4 > data.Length)
                return true;
        
            return BitConverter.ToInt32(data, XsensConstants.HeaderLength) != 0;
        }

        FrameData UpdateFrameMetadata(FrameData frame, int msgEntityID)
        {
            // Update from connection panel's stream toggle
            frame.enabled = StatusCollection[msgEntityID];
            
            // Update from latest metadata message
            if (MetaCollection.Count >= msgEntityID + 1)
                frame.metadata = MetaCollection[msgEntityID];
            if (ScaleMetaCollection.Count < msgEntityID + 1)
                return frame;
            
            // Update from latest scale data message
            frame.entityScaleData = ScaleMetaCollection[msgEntityID];
            try // propSegmentNames can briefly be null because of the if statement below, so we return the frame at this point
            {
                frame.entityScaleData.propSegmentNames.Clear();
            }
            catch (Exception)
            {
                return frame;
            }
            for (int i = 0; i < frame.NumProps; i++)
            {
                // The frame, metadata, and scale lists can get out of sync when a scene consists of a single actor with props plus and objects stream. 
                if (frame.entityScaleData.segmentNames.Count < XsensConstants.MvnBodySegmentCount + i)
                {
                    ScaleMetaCollection[msgEntityID] = new EntityScaleMetadata();
                    MetaCollection[msgEntityID] = new EntityMetadata();
                    return frame; 
                }
                string propName = frame.entityScaleData.segmentNames[XsensConstants.MvnBodySegmentCount + i];
                frame.entityScaleData.propSegmentNames.Add(propName);
            }
            frame.entityScaleData.fingerSegmentNames.Clear();
            if (frame.entityScaleData.isObjects)
                return frame;
            
            for (int i = 0; i < XsensConstants.MvnFingerSegmentCount; i++)
            {
                var index = XsensConstants.MvnBodySegmentCount + frame.NumProps + i;
                if (frame.entityScaleData.segmentNames == null || index >= frame.entityScaleData.segmentNames.Count)
                    continue;
                
                string segmentName = frame.entityScaleData.segmentNames[index];
                frame.entityScaleData.fingerSegmentNames.Add(segmentName);
            }
            return frame;
        }

        bool CheckEntityCountChanged(int msgEntityID)
        {
            return _prevMetaMsgID != msgEntityID - 1 && 
                   _prevMetaMsgID + 1 != _metaEntityCount;
        }
    }
}