﻿using System;
using System.IO;
using System.Linq;
using UnityEngine;
using UnityEditor;
using System.Collections.Generic;

[System.Serializable]
public struct BoneTransformData
{
    public HumanBodyBones HumanBone;
    public Vector3 Position;
    public Quaternion Rotation;
    public Vector3 Scale;

    public BoneTransformData(HumanBodyBones humanBone, Vector3 position, Quaternion rotation, Vector3 scale)
    {
        HumanBone = humanBone;
        Position = position;
        Rotation = rotation;
        Scale = scale;
    }
}

namespace Movella.Xsens.Editor
{
    public static class XsensAvatar
    {
        // Update these for the ui dropdowns
        public static GUIContent[] m_AvatarOverrides = {
            new GUIContent("No Override (Default)"), new GUIContent("Auto"), new GUIContent("Unity"), new GUIContent("UE5"), new GUIContent("UE4"), new GUIContent("Mixamo"),
            new GUIContent("HumanIK"), new GUIContent("Cinema 4D Biped"), new GUIContent("C4D Advanced Biped"), new GUIContent("AccuRig"), new GUIContent("Character Creator 4"), new GUIContent("Ready Player Me")
        };
        public static int[] m_AvatarOverrideValues = { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11 };

        // -- Create and set Avatar based on dropdown option --
        public static Avatar CreateAndSaveAvatar(GameObject rig, int type = 0)
        {
            if (rig == null)
            {
                Debug.LogError("Rig GameObject is null. Cannot create avatar.");
                return null;
            }

            if (type == 0) //None, retrieve and return the original default avatar
            {
                Animator animator = rig.GetComponent<Animator>();
                XsensDevice xsensComponent = rig.GetComponent<XsensDevice>();

                // Get the path of the original asset
                string assetPath = AssetDatabase.GetAssetPath(PrefabUtility.GetCorrespondingObjectFromSource(rig));
                if (string.IsNullOrEmpty(assetPath)) return null;

                // Load the ModelImporter to access the imported Avatar
                ModelImporter modelImporter = AssetImporter.GetAtPath(assetPath) as ModelImporter;
                if (modelImporter == null) return null;

                // Get the default avatar from the imported asset
                var defaultAvatar = AssetDatabase.LoadAssetAtPath<Avatar>(assetPath);
                if (!defaultAvatar.isValid || !defaultAvatar.isHuman)
                {
                    Debug.LogError("Failed to find default avatar.");
                    return null;
                }

                animator.avatar = defaultAvatar;

                // Toggle xsens device to accept avatar changes
                xsensComponent.enabled = false;
                xsensComponent.enabled = true;

                return defaultAvatar;
            }
            else if (type == 1) //Auto
            {
                return AutoAvatar(rig);
            }
            else if (type == 2) // Unity
            {
                return TemplateAvatar(rig, TPoseLibrary.UnityHumanoid);
            }
            else if (type == 3) // UE5
            {
                return TemplateAvatar(rig, TPoseLibrary.UE5);
            }
            else if (type == 4) // UE4
            {
                return TemplateAvatar(rig, TPoseLibrary.UE4);
            }
            else if (type == 5) // Mixamo
            {
                return TemplateAvatar(rig, TPoseLibrary.Mixamo);
            }
            else if (type == 6) // HumanIK
            {
                return TemplateAvatar(rig, TPoseLibrary.HumanIK);
            }
            else if (type == 7) // Cinema 4D
            {
                return TemplateAvatar(rig, TPoseLibrary.Cinema4DBiped);
            }
            else if (type == 8) // C4D Advanced Biped
            {
                return TemplateAvatar(rig, TPoseLibrary.Cinema4DAdvancedBiped);
            }
            else if (type == 9) // AccuRig
            {
                return TemplateAvatar(rig, TPoseLibrary.AccuRig);
            }
            else if (type == 10) // Character Creator 4
            {
                return TemplateAvatar(rig, TPoseLibrary.CharacterCreator4);
            }
            else if (type == 11) // Ready Player Me
            {
                return TemplateAvatar(rig, TPoseLibrary.ReadyPlayerMe);
            }

            return null;
        }

        // -- Template Avatar --
        public static Avatar TemplateAvatar(GameObject rig, Dictionary<string, BoneTransformData> tpose)
        {
            if (tpose == null) return null;

            Animator animator = rig.GetComponent<Animator>();
            XsensDevice xsensComponent = rig.GetComponent<XsensDevice>();
            string[] boneArray = rig.GetComponentsInChildren<Transform>().Select(t => t.name).ToArray();
            Transform[] bones = rig.GetComponentsInChildren<Transform>();

            HumanDescription humanDescription = new HumanDescription();
            List<HumanBone> humanBones = new List<HumanBone>();
            List<SkeletonBone> skeletonBones = new List<SkeletonBone>();
            List<string> mappedBones = new List<string>();

            foreach (KeyValuePair<string, BoneTransformData> boneMapping in tpose)
            {
                string boneName = null;

                // Check for exact match first
                if (boneArray.Contains(boneMapping.Key) && !mappedBones.Contains(boneMapping.Key) && boneName == null)
                {
                    boneName = boneMapping.Key;
                    mappedBones.Add(boneMapping.Key);
                }

                // Check for partial match
                foreach (Transform rigBone in bones)
                {
                    if (boneMapping.Value.HumanBone != HumanBodyBones.Hips && boneMapping.Value.HumanBone != HumanBodyBones.LastBone && mappedBones.Count() > 0)
                    {
                        string pelvis = mappedBones[0];
                        if (rigBone.name.Contains(boneMapping.Key) && !mappedBones.Contains(rigBone.name) && boneName == null && IsChildOf(rig, pelvis, rigBone.name))
                        {
                            mappedBones.Add(rigBone.name);
                            boneName = rigBone.name;
                        }
                    }
                    else
                    {
                        if (rigBone.name.Contains(boneMapping.Key) && !mappedBones.Contains(rigBone.name) && boneName == null)
                        {
                            mappedBones.Add(rigBone.name);
                            boneName = rigBone.name;
                        }
                    }
                }

                if (boneName == null)
                {
                    boneName = boneMapping.Key;
                    mappedBones.Add(boneMapping.Key);
                }

                if (boneArray.Contains(boneName))
                {
                    BoneTransformData transformData = boneMapping.Value;
                    HumanBodyBones boneType = transformData.HumanBone;

                    if (boneType != HumanBodyBones.LastBone)
                    {
                        // Define the human bone
                        HumanBone humanBone = new HumanBone
                        {
                            boneName = boneName,
                            humanName = TPoseLibrary.HumanBoneStrings[boneType],
                            limit = new HumanLimit { useDefaultValues = true }
                        };
                        humanBones.Add(humanBone);
                    }

                    // Define the skeleton bone
                    SkeletonBone skeletonBone = new SkeletonBone
                    {
                        name = boneName,
                        position = FindBoneTransform(rig, boneName).localPosition, // since this is specific to rig we dont want to use the template for these
                        rotation = transformData.Rotation,
                        scale = FindBoneTransform(rig, boneName).localScale // since this is specific to rig we dont want to use the template for these
                    };
                    skeletonBones.Add(skeletonBone);
                }
            }

            // Add in any extra bones
            foreach (Transform bone in bones)
            {
                if (!mappedBones.Contains(bone.name))
                {
                    // Define the skeleton bone
                    SkeletonBone skeletonBone = new SkeletonBone
                    {
                        name = bone.name,
                        position = bone.localPosition,
                        rotation = bone.localRotation,
                        scale = bone.localScale
                    };
                    skeletonBones.Add(skeletonBone);
                }
            }

            // Assign bones to HumanDescription
            humanDescription.human = humanBones.ToArray();
            humanDescription.skeleton = skeletonBones.ToArray();
            humanDescription.upperArmTwist = 0.5f;
            humanDescription.lowerArmTwist = 0.5f;
            humanDescription.upperLegTwist = 0.5f;
            humanDescription.lowerLegTwist = 0.5f;
            humanDescription.armStretch = 0.05f;
            humanDescription.legStretch = 0.05f;
            humanDescription.feetSpacing = 0.0f;
            humanDescription.hasTranslationDoF = false;

            var avatar = AvatarBuilder.BuildHumanAvatar(rig, humanDescription);
            if (!avatar.isValid || !avatar.isHuman)
            {
                Debug.LogError("Failed to create a valid humanoid avatar.");
                return null;
            }

            avatar.name = "XsensAvtr_" + rig.name; // TODO name based off template + rig name?
            animator.avatar = avatar;

            // Toggle xsens device to accept avatar changes
            xsensComponent.enabled = false;
            xsensComponent.enabled = true;

            // Save the Avatar as an asset
            /*
            string folderPath = "Assets/Xsens/";
            string avtrPath = folderPath + "XsensAvtr_" + rig.name + ".asset";

            if (!AssetDatabase.IsValidFolder(folderPath))
            {
                AssetDatabase.CreateFolder("Assets", "Xsens");
            }

            AssetDatabase.CreateAsset(avatar, avtrPath);
            AssetDatabase.SaveAssets();
            AssetDatabase.Refresh();
            */

            return avatar;
        }

        // -- Auto Avatar --
        public static Avatar AutoAvatar(GameObject rig)
        {
            XsensDevice xsensComponent = rig.GetComponent<XsensDevice>();
            Transform[] bones = rig.GetComponentsInChildren<Transform>();
            Animator animator = rig.GetComponent<Animator>();
            string[] boneArray = rig.GetComponentsInChildren<Transform>().Select(t => t.name).ToArray();

            // Reset to old Tpose in order to calculate new positions
            ResetTPoseFromAvatar(rig);

            Dictionary<string, Quaternion> foot_rot = new Dictionary<string, Quaternion>();

            HumanDescription humanDescription = new HumanDescription();
            List<HumanBone> humanBones = new List<HumanBone>();
            List<SkeletonBone> skeletonBones = new List<SkeletonBone>();

            // Human Bones

            // Get proper mapping for human body bones
            List<string> mappedBones = new List<string>();
            foreach (KeyValuePair<HumanBodyBones, List<string>> boneMapping in TPoseLibrary.AutoTemplate)
            {
                string boneName = null;

                // Check for exact match first
                foreach (string potentialName in boneMapping.Value)
                {
                    if (boneArray.Contains(potentialName) && !mappedBones.Contains(potentialName) && boneName == null)
                    {
                        boneName = potentialName;
                        mappedBones.Add(potentialName);
                    }
                }

                // Check for partial match
                foreach (string potentialName in boneMapping.Value)
                {
                    if (boneName == null)
                    {
                        foreach (Transform bone in bones)
                        {
                            // If human bone is not pelvis or last bone check if it has pelvis as its parent?
                            if (boneMapping.Key != HumanBodyBones.Hips && boneMapping.Key != HumanBodyBones.LastBone && mappedBones.Count() > 0)
                            {
                                string pelvis = mappedBones[0];
                                if (bone.name.ToLower().Contains(potentialName) && !mappedBones.Contains(bone.name) && boneName == null && IsChildOf(rig, pelvis, bone.name))
                                {
                                    boneName = bone.name;
                                    mappedBones.Add(bone.name);
                                }
                            }
                            else
                            {
                                if (bone.name.ToLower().Contains(potentialName) && !mappedBones.Contains(bone.name) && boneName == null)
                                {
                                    boneName = bone.name;
                                    mappedBones.Add(bone.name);
                                }
                            }
                        }
                    }
                }
                if (boneName != null)
                {
                    // Define the human bone
                    HumanBone humanBone = new HumanBone
                    {
                        boneName = boneName,
                        humanName = TPoseLibrary.HumanBoneStrings[boneMapping.Key],
                        limit = new HumanLimit { useDefaultValues = true }
                    };
                    humanBones.Add(humanBone);
                }
            }

            // Skeleton Bones

            // Save the original foot rotational values to be set later.
            foreach (Transform bone in bones)
            {
                if (GetIsOfBone(bone.name, humanBones, new string[] { "LeftFoot", "RightFoot" }))
                {
                    foot_rot[bone.name] = bone.rotation;
                }
            }

            // Get proper rotation for skeleton bones
            foreach (Transform bone in bones)
            {
                // Initial offsets upperarms
                if (GetIsOfBone(bone.name, humanBones, new string[] { "LeftUpperArm" }))
                {
                    bone.rotation = Quaternion.Euler(0, 0, -20) * bone.rotation;
                }
                else if (GetIsOfBone(bone.name, humanBones, new string[] { "RightUpperArm" }))
                {
                    bone.rotation = Quaternion.Euler(0, 0, 20) * bone.rotation;
                }
                // Initial offsets elbows
                else if (GetIsOfBone(bone.name, humanBones, new string[] { "LeftLowerArm" }))
                {
                    bone.rotation = Quaternion.Euler(0, -20, 0) * bone.rotation;
                }
                else if (GetIsOfBone(bone.name, humanBones, new string[] { "RightLowerArm" }))
                {
                    bone.rotation = Quaternion.Euler(0, 20, 0) * bone.rotation;
                }
                // Initial offsets thumbs
                else if (GetIsOfBone(bone.name, humanBones, new string[] { "Left Thumb Proximal" }))
                {
                    bone.rotation = Quaternion.Euler(0, -20, 0) * bone.rotation;
                    bone.rotation = Quaternion.Euler(-44, 0, 0) * bone.rotation;
                }
                else if (GetIsOfBone(bone.name, humanBones, new string[] { "Right Thumb Proximal" }))
                {
                    bone.rotation = Quaternion.Euler(0, 20, 0) * bone.rotation;
                    bone.rotation = Quaternion.Euler(-44, 0, 0) * bone.rotation;
                }
                // Initial offsets fingers
                else if (GetIsOfBone(bone.name, humanBones, new string[] { 
                    "Left Index Proximal", "Left Index Intermediate", "Left Index Distal",
                    "Left Middle Proximal", "Left Middle Intermediate", "Left Middle Distal",
                    "Left Ring Proximal", "Left Ring Intermediate", "Left Ring Distal",
                    "Left Little Proximal", "Left Little Intermediate", "Left Little Distal" 
                }))
                {
                    bone.rotation = Quaternion.Euler(0, 0, -20) * bone.rotation;
                }
                else if (GetIsOfBone(bone.name, humanBones, new string[] { 
                    "Right Index Proximal", "Right Index Intermediate", "Right Index Distal",
                    "Right Middle Proximal", "Right Middle Intermediate", "Right Middle Distal",
                    "Right Ring Proximal", "Right Ring Intermediate", "Right Ring Distal",
                    "Right Little Proximal", "Right Little Intermediate", "Right Little Distal" 
                }))
                {
                    bone.rotation = Quaternion.Euler(0, 0, 20) * bone.rotation;
                }
                // Snap all these bones to nearest 90
                if (GetIsOfBone(bone.name, humanBones, new string[] { 
                    "LeftUpperArm", "LeftLowerArm", "LeftHand", "LeftUpperLeg", "LeftLowerLeg",
                    "Left Thumb Proximal", "Left Thumb Intermediate", "Left Thumb Distal",
                    "Left Index Proximal", "Left Index Intermediate", "Left Index Distal",
                    "Left Middle Proximal", "Left Middle Intermediate", "Left Middle Distal",
                    "Left Ring Proximal", "Left Ring Intermediate", "Left Ring Distal",
                    "Left Little Proximal", "Left Little Intermediate", "Left Little Distal",
                    "RightUpperArm", "RightLowerArm", "RightHand", "RightUpperLeg", "RightLowerLeg",
                    "Right Thumb Proximal", "Right Thumb Intermediate", "Right Thumb Distal",
                    "Right Index Proximal", "Right Index Intermediate", "Right Index Distal",
                    "Right Middle Proximal", "Right Middle Intermediate", "Right Middle Distal",
                    "Right Ring Proximal", "Right Ring Intermediate", "Right Ring Distal",
                    "Right Little Proximal", "Right Little Intermediate", "Right Little Distal"
                }))
                {
                    // Round each component to the nearest 90 degrees
                    Vector3 eulerAngles = bone.rotation.eulerAngles;
                    eulerAngles.x = Mathf.Round(eulerAngles.x / 90f) * 90f;
                    eulerAngles.y = Mathf.Round(eulerAngles.y / 90f) * 90f;
                    eulerAngles.z = Mathf.Round(eulerAngles.z / 90f) * 90f;

                    // Convert back to quaternion
                    Quaternion roundedRotation = Quaternion.Euler(eulerAngles);
                    bone.rotation = roundedRotation;
                }
                // Reset feet rotation to what they were
                if (GetIsOfBone(bone.name, humanBones, new string[] { "LeftFoot", "RightFoot" }))
                {
                    bone.rotation = foot_rot[bone.name];
                }

                // Define and add new skeleton bone
                SkeletonBone skeletonBone = new SkeletonBone
                {
                    name = bone.name,
                    position = bone.localPosition,
                    rotation = bone.localRotation,
                    scale = bone.localScale
                };
                skeletonBones.Add(skeletonBone);
            }

            // Assign bones to HumanDescription
            humanDescription.human = humanBones.ToArray();
            humanDescription.skeleton = skeletonBones.ToArray();
            humanDescription.upperArmTwist = 0.5f;
            humanDescription.lowerArmTwist = 0.5f;
            humanDescription.upperLegTwist = 0.5f;
            humanDescription.lowerLegTwist = 0.5f;
            humanDescription.armStretch = 0.05f;
            humanDescription.legStretch = 0.05f;
            humanDescription.feetSpacing = 0.0f;
            humanDescription.hasTranslationDoF = false;

            var avatar = AvatarBuilder.BuildHumanAvatar(rig, humanDescription);
            if (!avatar.isValid || !avatar.isHuman)
            {
                Debug.LogError("Failed to create a valid auto avatar.");
                return null;
            }

            avatar.name = "XsensAutoAvtr_" + rig.name; // TODO name based off template + rig name?
            animator.avatar = avatar;

            // Toggle xsens device to accept avatar changes
            xsensComponent.enabled = false;
            xsensComponent.enabled = true;

            return avatar;
        }

        // -- Utility functions -
        public static bool GetIsOfBone(string boneName, List<HumanBone> humanBones, string[] boneNamesArray)
        {
            foreach (HumanBone humanBone in humanBones)
            {
                if (humanBone.boneName == boneName && boneNamesArray.Contains(humanBone.humanName))
                {
                    return true;
                }
            }
            return false;
        }

        public static void ResetTPoseFromAvatar(GameObject rig)
        {
            XsensDevice xsensComponent = rig.GetComponent<XsensDevice>();
            Transform[] bones = rig.GetComponentsInChildren<Transform>();
            Animator animator = rig.GetComponent<Animator>();

            // Save current avatar to use for zero values
            Avatar oldAvtr = animator.avatar;

            // Clear current avatar and refresh component
            animator.avatar = null;
            xsensComponent.enabled = false;

            // Zero rotations to current avatar if one exists
            if (oldAvtr != null)
            {
                foreach (Transform bone in bones)
                {
                    foreach (SkeletonBone skelBone in oldAvtr.humanDescription.skeleton)
                    {
                        if (bone.name == skelBone.name)
                        {
                            bone.localRotation = skelBone.rotation;
                            bone.localPosition = skelBone.position;
                        }
                    }
                }
            }
            xsensComponent.enabled = true;
        }

        public static Transform FindBoneTransform(GameObject rootObject, string boneName)
        {
            if (rootObject == null) { return null; }

            Transform boneTransform = FindChildByName(rootObject.transform, boneName);

            if (boneTransform == null) { return null; }

            return boneTransform;
        }

        private static Transform FindChildByName(Transform parent, string name)
        {
            foreach (Transform child in parent.GetComponentsInChildren<Transform>())
            {
                if (child.name == name)
                {
                    return child;
                }
            }
            return null;
        }

        private static bool IsChildOf(GameObject root, string parentName, string childName)
        {
            if (root == null) return false;

            Transform parent = FindBone(root.transform, parentName);
            Transform child = FindBone(root.transform, childName);

            if (parent == null || child == null) return false;

            return child.IsChildOf(parent);
        }

        public static Transform FindBone(Transform root, string boneName)
        {
            foreach (Transform child in root.GetComponentsInChildren<Transform>())
            {
                if (child.name == boneName)
                    return child;
            }
            return null;
        }

    }
}
