﻿using System;
using System.Collections.Generic;
using System.Threading.Tasks;
using Codice.CM.Common.Serialization;
using UnityEngine;
using UnityEditor;

namespace Movella.Xsens.Editor
{
    public class XsensLiveStreamEditor : EditorWindow
    {
        [Serializable]
        public class StreamTargetState
        {
            public bool Enable { get; set; }
            public int SelectedStreamChannelIndex { get; set; }
            public int SelectedPropChannelIndex { get; set; }
            public int SelectedPropParentIndex { get; set; }
            public string[] StreamChannels { get; set; }
            public string[] PropChannels { get; set; }
            public GameObject TargetRig { get; set; }
            public GameObject PropChild { get; set; }
            public GameObject PropParent { get; set; }

            public StreamTargetState(SerializedProperty element)
            {
                Enable = element.FindPropertyRelative("enable").boolValue;
                SelectedStreamChannelIndex = element.FindPropertyRelative("selectedStreamChannelIndex").intValue;
                SelectedPropChannelIndex = element.FindPropertyRelative("selectedPropChannelIndex").intValue;
                SelectedPropParentIndex = element.FindPropertyRelative("selectedPropParentIndex").intValue;

                var streamChannels = element.FindPropertyRelative("streamChannels");
                StreamChannels = new string[streamChannels.arraySize];
                for (int i = 0; i < streamChannels.arraySize; i++)
                {
                    StreamChannels[i] = streamChannels.GetArrayElementAtIndex(i).stringValue;
                }

                var propChannels = element.FindPropertyRelative("propChannels");
                PropChannels = new string[propChannels.arraySize];
                for (int i = 0; i < propChannels.arraySize; i++)
                {
                    PropChannels[i] = propChannels.GetArrayElementAtIndex(i).stringValue;
                }

                TargetRig = element.FindPropertyRelative("targetRig").objectReferenceValue as GameObject;
                PropChild = element.FindPropertyRelative("propChild").objectReferenceValue as GameObject;
                PropParent = element.FindPropertyRelative("propParent").objectReferenceValue as GameObject;
            }

            public void RestoreToElement(SerializedProperty element)
            {
                element.FindPropertyRelative("enable").boolValue = Enable;
                element.FindPropertyRelative("selectedStreamChannelIndex").intValue = SelectedStreamChannelIndex;
                element.FindPropertyRelative("selectedPropChannelIndex").intValue = SelectedPropChannelIndex;
                element.FindPropertyRelative("selectedPropParentIndex").intValue = SelectedPropParentIndex;

                var streamChannels = element.FindPropertyRelative("streamChannels");
                streamChannels.arraySize = StreamChannels.Length;
                for (int i = 0; i < StreamChannels.Length; i++)
                {
                    streamChannels.GetArrayElementAtIndex(i).stringValue = StreamChannels[i];
                }

                var propChannels = element.FindPropertyRelative("propChannels");
                propChannels.arraySize = PropChannels.Length;
                for (int i = 0; i < PropChannels.Length; i++)
                {
                    propChannels.GetArrayElementAtIndex(i).stringValue = PropChannels[i];
                }

                element.FindPropertyRelative("targetRig").objectReferenceValue = TargetRig;
                element.FindPropertyRelative("propChild").objectReferenceValue = PropChild;
                element.FindPropertyRelative("propParent").objectReferenceValue = PropParent;
            }

            public override string ToString()
            {
                var sb = new System.Text.StringBuilder();
                sb.AppendLine("Stream Target State:");
                sb.AppendLine($"Enable: {Enable}");
                sb.AppendLine($"Selected Stream Channel Index: {SelectedStreamChannelIndex}");
                sb.AppendLine($"Selected Prop Channel Index: {SelectedPropChannelIndex}");
                sb.AppendLine($"Selected Prop Parent Index: {SelectedPropParentIndex}");
                sb.AppendLine($"Stream Channels: [{string.Join(", ", StreamChannels)}]");
                sb.AppendLine($"Prop Channels: [{string.Join(", ", PropChannels)}]");
                sb.AppendLine($"Target Rig: {(TargetRig != null ? TargetRig.name : "null")}");
                sb.AppendLine($"Prop Child: {(PropChild != null ? PropChild.name : "null")}");
                sb.AppendLine($"Prop Parent: {(PropParent != null ? PropParent.name : "null")}");
                return sb.ToString();
            }
        }
        // Scroll View
        private Vector2 _scrollPosition;
        
        // Fields
        private XsensRemoteControl _mRemoteControl;
        private PlaybackTimeline _timeline;
        private AddPrefabTargetDialog _prefabDialog;
        private XsensConnection _connection;
        
        private List<StreamTargetState> _lastKnownState = new List<StreamTargetState>();
        private bool _isSceneSwitching;
        
        // Stream Panel Fields
        private bool _showStreamPanel = true;
        private string _hostAddress = "localhost";
        private int _listenPort = 9763;
        private bool _isListening;

        // Control Panel Fields
        private bool _showControlPanel = true;
        private bool _isTakeRecorderTriggerEnabled;
        private int _remoteControlPort = 6004;
        private bool _isPlayback;
        private bool _isLooping;
        private string _takeName = "New Session";
        private int _takeNumber = 1;
        private bool _isRecording;
        private string[] _takeRecorderFrameRateOptions;

        // Stream Targets Panel Fields
        private List<string> _actors;
        private GameObject targetRig;
        private bool _showStreamTargetsPanel = true;
        private bool _showDevicePanel = true;
        private bool _showPropsPanel = false;

        // GUI Content
        private GUIContent _listenPortElementContent;
        private GUIContent _connectionsButtonContent;
        private GUIContent _diagnosticsButtonContent;
        private GUIContent _hostAddressElementContent;
        private GUIContent _remoteControlPortElementContent;
        private GUIContent _firstFrameIcon;
        private GUIContent _lastFrameIcon;
        private GUIContent _playIcon;
        private GUIContent _nextFrameIcon;
        private GUIContent _previousFrameIcon;
        private GUIContent _playLoopOffIcon;
        private GUIContent _moveToOriginIcon;
        private GUIContent _resetAxisIcon;
        private GUIContent _recordIcon;
        private GUIContent _triggerTakeRecorderToggleContent;
        private GUIContent _openTakeRecorderButtonContent;
        
        // Colors
        private Color _toggleColor;
        private Color _toggleRecordColor;
        
        // Constants
        private const string IconPath = "Packages/com.movella.xsens/Editor/Icons";
        
        [MenuItem("Window/Xsens Live Stream")]

        // -------------------------------------------------------------------------------------------------------------
        // View
        public static void ShowWindow()
        {
            GetWindow<XsensLiveStreamEditor>("Xsens Live Stream");
        }
        
        private void OnEnable()
        {
            _mRemoteControl = new XsensRemoteControl();
            _timeline = new PlaybackTimeline();
            _timeline.OnEnable();
            _timeline.OnFrameUpdated += HandleUserScrubPlayhead;

            InitializeGuiContent();
            InitializeColors();
        }
        
        
        private void OnDisable()
        {
            _timeline.OnFrameUpdated -= HandleUserScrubPlayhead;
        }

        private void InitializeGuiContent()
        {
            _connectionsButtonContent = new GUIContent("Connections", "Open the Unity Live Capture connections window");
            _diagnosticsButtonContent = new GUIContent("Diagnostics", "Open the Xsens stream diagnostics window");
            _listenPortElementContent = new GUIContent("Listen Port", "The local port that listens for incoming Xsens data");
            _hostAddressElementContent = new GUIContent("Host Address", "The IP address of the device running the MVN session");
            _remoteControlPortElementContent = new GUIContent("Remote Control Port", "The Remote Control listen port of the MVN session");
            _triggerTakeRecorderToggleContent = new GUIContent("Trigger Take Recorder", "If enabled, the record button will also trigger Unity's Take Recorder");
            _openTakeRecorderButtonContent = new GUIContent("           Take Recorder           ", "Open Unity's Take Recoder Panel");
            _firstFrameIcon = new GUIContent(EditorGUIUtility.IconContent("Animation.FirstKey").image, "Move MVN playhead to the first frame");
            _previousFrameIcon = new GUIContent(EditorGUIUtility.IconContent( "Animation.PrevKey").image, "Move MVN playhead back one frame");
            _playIcon = new GUIContent(EditorGUIUtility.IconContent("PlayButton").image, "Toggle MVN Playback/Pause");
            _nextFrameIcon = new GUIContent(EditorGUIUtility.IconContent("Animation.NextKey").image, "Move MVN playhead forward one frame");
            _lastFrameIcon = new GUIContent(EditorGUIUtility.IconContent("Animation.LastKey").image, "Move MVN playhead to the final frame");
            _playLoopOffIcon = new GUIContent(EditorGUIUtility.IconContent("PlayLoopOff").image, "Toggle MVN playback looping");
            _moveToOriginIcon = new GUIContent((Texture)AssetDatabase.LoadAssetAtPath($"{IconPath}/MoveToOriginIcon.PNG", typeof(Texture)), "Move MVN character to the origin");
            _resetAxisIcon = new GUIContent((Texture)AssetDatabase.LoadAssetAtPath($"{IconPath}/AxisResetIcon.PNG", typeof(Texture)), "Reset the MVN X-axis in to align with the current direction the actor is facing");
            _recordIcon = new GUIContent(EditorGUIUtility.IconContent("Animation.Record").image, "Start Recording with Unity Take Recorder");
        }

        private void InitializeColors()
        {
            _toggleColor = new Color(100f / 255f, 180f / 255f, 255f / 255f);
            _toggleRecordColor = new Color(255f / 255f, 0f / 255f, 0f / 255f);
        }

        private void OnGUI()
        {
            _scrollPosition = EditorGUILayout.BeginScrollView(_scrollPosition);

            DrawStreamPanel();
            GUILayout.Box("", GUILayout.ExpandWidth(true), GUILayout.Height(5));
            DrawControlPanel();
            GUILayout.Box("", GUILayout.ExpandWidth(true), GUILayout.Height(5));
            DrawStreamTargetsPanel();

            EditorGUILayout.EndScrollView();
        }

        private void DrawStreamPanel()
        {
            _showStreamPanel = EditorGUILayout.Foldout(_showStreamPanel, "Xsens Live Streaming");
            if (!_showStreamPanel) return;
            _listenPort = EditorGUILayout.IntField(_listenPortElementContent, _listenPort);
            var streamButtonText = _isListening ? "Pause Stream" : "Start Stream";
            GUI.backgroundColor = _isListening ? _toggleColor : Color.white;
            if (GUILayout.Button(streamButtonText))
            {
                OnStartStreamClicked();
            }
            GUI.backgroundColor = Color.white;
                
            GUILayout.BeginHorizontal();
            if (GUILayout.Button(_connectionsButtonContent))
            {
                OnConnectionsClicked();
            }
                
            if (GUILayout.Button(_diagnosticsButtonContent))
            {
                OnDiagnosticsClicked();
            }
                
            GUILayout.EndHorizontal();
        }

        private void DrawControlPanel()
        {
            _showControlPanel = EditorGUILayout.Foldout(_showControlPanel, "Xsens Control Panel");
            if (!_showControlPanel) return;
            GUILayout.Label("Playback Controls");
            _hostAddress = EditorGUILayout.TextField(_hostAddressElementContent, _hostAddress);
            _remoteControlPort = EditorGUILayout.IntField(_remoteControlPortElementContent, _remoteControlPort);
            _timeline.OnGUI();

            GUILayout.BeginHorizontal();
            if (GUILayout.Button(_firstFrameIcon))
            {
                SetPlayheadPosition("first");
            }

            if (GUILayout.Button(_previousFrameIcon))
            {
                SetPlayheadPosition("previous");
            }

            GUI.backgroundColor = _isPlayback ? _toggleColor : Color.white;
            if (GUILayout.Button(_playIcon))
            {
                OnTogglePlaybackClicked();
            }

            GUI.backgroundColor = Color.white;
            if (GUILayout.Button(_nextFrameIcon))
            {
                SetPlayheadPosition("next");
            }

            if (GUILayout.Button(_lastFrameIcon))
            {
                SetPlayheadPosition("last");

            }

            GUI.backgroundColor = _isLooping ? _toggleColor : Color.white;
            if (GUILayout.Button(_playLoopOffIcon))
            {
                OnToggleLoopClicked();
            }

            GUI.backgroundColor = Color.white;
            GUILayout.EndHorizontal();

            GUILayout.Label("Live Session Controls");

            GUILayout.BeginHorizontal();
            GUILayout.Label("Take");
            _takeName = EditorGUILayout.TextField("", _takeName);
            _takeNumber = EditorGUILayout.IntField("", _takeNumber, GUILayout.Width(30));
            GUILayout.EndHorizontal();

            GUILayout.BeginHorizontal();
            GUI.backgroundColor = Color.white;
            if (GUILayout.Button(_moveToOriginIcon))
            {
                OnActivateLiveSessionFeatureClicked("moveToOrigin");
            }

            if (GUILayout.Button(_resetAxisIcon))
            {
                OnActivateLiveSessionFeatureClicked("resetAxis");
            }

            GUILayout.EndHorizontal();

            // Record button
            var recordButtonText = _isRecording ? "Stop MVN Recording" : "Start MVN Recording";
            GUI.backgroundColor = _isRecording ? _toggleRecordColor : Color.white;
            var recordButtonContent = new GUIContent(recordButtonText, _recordIcon.image, "Start/Stop MVN Recording");
            if (GUILayout.Button(recordButtonContent))
            {
                OnStartRecordClicked();
            }
            GUI.backgroundColor = Color.white;
            
            
            GUILayout.BeginHorizontal();
            // Take recorder toggle
            _isTakeRecorderTriggerEnabled = EditorGUILayout.Toggle(_triggerTakeRecorderToggleContent, _isTakeRecorderTriggerEnabled);
            
            // Open Take recorder

            if (GUILayout.Button(_openTakeRecorderButtonContent))
            {
                OnOpenTakeRecorderButtonClicked();
            }
            GUILayout.EndHorizontal();
        }

        private void DrawStreamTargetsPanel()
        {
            _showStreamTargetsPanel = EditorGUILayout.Foldout(_showStreamTargetsPanel, "Stream Targets");

            // If the dropdown is not visible, the list will not be drawn
            if (!_showStreamTargetsPanel) return;
            EditorGUILayout.Space();

            EditorGUIUtility.wideMode = true;

            EditorGUI.indentLevel++;

            // -- Target Rig
            targetRig = (GameObject)EditorGUILayout.ObjectField(
                new GUIContent("Target", "Select game object to stream to."),
                targetRig, typeof(GameObject), true
            );
            EditorGUILayout.Space(1);

            EditorGUI.indentLevel--;

            XsensDevice xsensComponent = null;
            XsensObjectDevice xsensObjectComponent = null;
            int _xsensSourceID = 0; // selected source ID

            // Get or add proper Xsens component and set stream ID
            if (targetRig != null)
            {
                Animator animator = targetRig.GetComponent<Animator>();
                // -- Xsens Source
                _actors = new List<string>();
                if (_connection != null && _connection.Client != null)
                {
                    var metadata = _connection.Client.MetaCollection;
                    var scalemetadata = _connection.Client.ScaleMetaCollection;
                    for (int i = 0; i < metadata.Count; i++)
                    {
                        var name =metadata[i].entityName;
                        if (string.IsNullOrEmpty(name))
                            continue;
                        if (_actors.Contains(name))
                        {
                            name = $"{name}:{i}";
                        }
                        _actors.Add(name);
                    }

                    // Attempt to get xsens device from rig first and extract xsens source ID to set here
                    if (targetRig.GetComponent<XsensDevice>())
                    {
                        xsensComponent = targetRig.GetComponent<XsensDevice>();
                        _xsensSourceID = xsensComponent.CharacterID;
                    }
                    if (targetRig.GetComponent<XsensObjectDevice>())
                    {
                        xsensObjectComponent = targetRig.GetComponent<XsensObjectDevice>();
                        _xsensSourceID = xsensObjectComponent.StreamID;
                    }

                    EditorGUI.indentLevel++;
                    _xsensSourceID = EditorGUILayout.Popup(
                        new GUIContent("Xsens Source", "Select stream subject from Xsens."), 
                        _xsensSourceID, _actors.ToArray()
                    );
                    EditorGUILayout.Space();
                    EditorGUI.indentLevel--;

                    // Get or add proper xsens device, actor or object
                    if (scalemetadata.Count > _xsensSourceID)
                    {
                        bool isObj = scalemetadata[_xsensSourceID].isObjects;
                        if (!isObj) // actor
                        {
                            if (targetRig.GetComponent<XsensObjectDevice>())
                            {
                                DestroyImmediate(targetRig.GetComponent<XsensObjectDevice>());
                            }
                            if (targetRig.GetComponent<XsensDevice>() == null)
                            {
                                xsensComponent = targetRig.AddComponent<XsensDevice>();
                            }
                            else if (targetRig.GetComponent<XsensDevice>())
                            {
                                xsensComponent = targetRig.GetComponent<XsensDevice>();
                            }
                            // set device stream ID
                            SerializedObject xsensSerializedObject = new SerializedObject(xsensComponent);
                            xsensSerializedObject.Update();
                            SerializedProperty charID = xsensSerializedObject.FindProperty("m_CharacterID");
                            charID.intValue = _xsensSourceID;
                            xsensSerializedObject.ApplyModifiedProperties();

                        }
                        else // object
                        {
                            if (targetRig.GetComponent<XsensDevice>())
                            {
                                DestroyImmediate(targetRig.GetComponent<XsensDevice>());
                            }
                            if (targetRig.GetComponent<XsensObjectDevice>() == null)
                            {
                                xsensObjectComponent = targetRig.AddComponent<XsensObjectDevice>();
                            }
                            else if (targetRig.GetComponent<XsensObjectDevice>())
                            {
                                xsensObjectComponent = targetRig.GetComponent<XsensObjectDevice>();
                            }
                            // set device stream ID
                            SerializedObject xsensSerializedObject = new SerializedObject(xsensObjectComponent);
                            xsensSerializedObject.Update();
                            SerializedProperty streamID = xsensSerializedObject.FindProperty("m_StreamID");
                            streamID.intValue = _xsensSourceID;
                            xsensSerializedObject.ApplyModifiedProperties();
                        }
                    }
                }

                else // no mvn connection
                {
                    EditorGUI.indentLevel++;
                    _xsensSourceID = EditorGUILayout.Popup(
                        new GUIContent("Xsens Source", "Select stream subject from Xsens."), 
                        _xsensSourceID, _actors.ToArray()
                    );
                    EditorGUILayout.Space(1);
                    EditorGUILayout.HelpBox("No Xsens Connection.", MessageType.Info);
                    EditorGUILayout.Space(1);
                    EditorGUI.indentLevel--;
                }

                // -- Xsens Actor Device Properties
                if (xsensComponent != null)
                {
                    SerializedObject xsensSerializedObject = new SerializedObject(xsensComponent);
                    xsensSerializedObject.Update();

                    SerializedObject serializedAnimator = new SerializedObject(animator);
                    serializedAnimator.Update();

                    xsensComponent.enabled = true;

                    EditorGUILayout.Space();
                    _showDevicePanel = EditorGUILayout.InspectorTitlebar(_showDevicePanel, xsensComponent);
                    EditorGUILayout.Space();

                    if (_showDevicePanel)
                    {
                        EditorGUI.indentLevel++;
                        // -- Animator
                        //EditorGUI.BeginDisabledGroup(true); // Disable UI elements
                        SerializedProperty anim = xsensSerializedObject.FindProperty("m_Animator");
                        EditorGUILayout.PropertyField(anim, true);
                        EditorGUILayout.Space(1);
                        //EditorGUI.EndDisabledGroup();

                        //SerializedProperty avtr = serializedAnimator.FindProperty("m_Avatar"); // do we want to show the current avatar
                        //EditorGUILayout.PropertyField(avtr, true);
                        //EditorGUILayout.Space();

                        // -- Avatar Override
                        SerializedProperty avatarOverride = xsensSerializedObject.FindProperty("m_AvatarOverride");
                        int previousValue = avatarOverride.intValue;
                        int newValue = EditorGUILayout.IntPopup(
                            new GUIContent(avatarOverride.displayName, 
                            "Selecting an override will create a new avatar from a template or auto process " +
                            "(Avatars are used for bone mappings and t-pose values)"), 
                            previousValue, XsensAvatar.m_AvatarOverrides, XsensAvatar.m_AvatarOverrideValues
                        );
                        if (previousValue != newValue)
                        {
                            Avatar newAvtr = XsensAvatar.CreateAndSaveAvatar(targetRig, newValue);
                            if (newAvtr == null)
                            {
                                XsensAvatar.ResetTPoseFromAvatar(targetRig);
                            }
                            avatarOverride.intValue = newValue;
                            xsensSerializedObject.ApplyModifiedProperties();
                        }

                        EditorGUILayout.Space(1);

                        // -- Apply Root Motion
                        EditorGUILayout.PropertyField(
                            serializedAnimator.FindProperty("m_ApplyRootMotion"),
                            new GUIContent("Apply Root Motion", "Toggle lateral movement from the stream data.")
                        );
                        EditorGUILayout.Space(1);

                        if (animator == null || !animator.avatar || !animator.avatar.isHuman)
                        {
                            EditorGUILayout.Space();
                            EditorGUILayout.HelpBox("Avatar is required for streaming. Create one manually or select a valid override", MessageType.Error);
                        }

                        EditorGUILayout.PropertyField( xsensSerializedObject.FindProperty("m_Channels") );
                        EditorGUILayout.Space(1);

                        EditorGUILayout.PropertyField( xsensSerializedObject.FindProperty("m_Recorder") );
                        EditorGUILayout.Space(1);

                        EditorGUILayout.PropertyField( xsensSerializedObject.FindProperty("m_positionOffset") );
                        EditorGUILayout.Space(1);

                        EditorGUILayout.PropertyField( xsensSerializedObject.FindProperty("m_scale") );
                        EditorGUILayout.Space(2);

                        // Display props property separate with a better UI
                        SerializedProperty props = xsensSerializedObject.FindProperty("m_Props");
                        if (props != null)
                        {
                            _showPropsPanel = EditorGUILayout.Foldout(_showPropsPanel, "Props");
                            if (_showPropsPanel)
                            {
                                EditorGUI.indentLevel++;
                                for (int i = 0; i < props.arraySize; i++)
                                {
                                    var propName = "";
                                    if (_connection?.Client?.ScaleMetaCollection != null)
                                    {
                                        int id = xsensSerializedObject.FindProperty("m_CharacterID").intValue;
                                        if (id < _connection.Client.ScaleMetaCollection.Count)
                                        {
                                            var propNames = _connection.Client.ScaleMetaCollection[id].propSegmentNames;
                                            if (propNames != null && i < propNames.Count)
                                                propName = $" - {propNames[i]}";
                                        }
                                    }
                                    var prop = props.GetArrayElementAtIndex(i);
                                    EditorGUILayout.PropertyField(prop, new GUIContent($"Prop {i + 1}{propName}"));
                                }
                                EditorGUI.indentLevel--;
                            }
                        }
                        EditorGUILayout.Space(1);

                        xsensSerializedObject.ApplyModifiedProperties();
                        serializedAnimator.ApplyModifiedProperties();
                        EditorGUI.indentLevel--;
                    }
                }

                // -- Xsens Object Device Properties
                if (xsensObjectComponent != null)
                {
                    SerializedObject xsensSerializedObject = new SerializedObject(xsensObjectComponent);
                    xsensSerializedObject.Update();

                    xsensObjectComponent.enabled = true;
                    EditorGUILayout.Space();
                    _showDevicePanel = EditorGUILayout.InspectorTitlebar(_showDevicePanel, xsensObjectComponent);
                    EditorGUILayout.Space();
                    var streamID = xsensSerializedObject.FindProperty("m_StreamID");

                    if (_showDevicePanel)
                    {
                        EditorGUI.indentLevel++;
                        DrawObjectIDList(streamID.intValue, xsensSerializedObject.FindProperty("m_ObjectID"));
                        EditorGUILayout.Space(1);

                        EditorGUILayout.PropertyField(xsensSerializedObject.FindProperty("m_Animator"));
                        EditorGUILayout.Space(1);

                        EditorGUILayout.PropertyField(xsensSerializedObject.FindProperty("m_Recorder"));
                        EditorGUILayout.Space(1);

                        EditorGUILayout.PropertyField(xsensSerializedObject.FindProperty("m_positionOffset"));
                        EditorGUILayout.Space(1);

                        EditorGUILayout.PropertyField(xsensSerializedObject.FindProperty("m_scale"));
                        EditorGUILayout.Space(1);

                        xsensSerializedObject.ApplyModifiedProperties();
                        EditorGUI.indentLevel--;
                    }
                }
                EditorGUILayout.Space();

                // -- Delete stream from target Button
                EditorGUILayout.Space(5);
                GUILayoutOption[] buttonOptions = { GUILayout.Width(240), GUILayout.Height(24) };
                using (new GUILayout.HorizontalScope())
                {
                    GUILayout.FlexibleSpace();
                    if (GUILayout.Button("Delete Stream From Target", buttonOptions))
                    {
                        // remove all xsens devices and remove target rig selection
                        var rig = targetRig;
                        targetRig = null;
                        if (rig.GetComponent<XsensDevice>())
                        {
                            DestroyImmediate(rig.GetComponent<XsensDevice>());
                        }
                        if (rig.GetComponent<XsensObjectDevice>())
                        {
                            DestroyImmediate(rig.GetComponent<XsensObjectDevice>());
                        }
                    }
                    GUILayout.FlexibleSpace();
                }

                // Print tpose and mappings - Dev Only
                //EditorGUILayout.Space(5);
                //if (GUILayout.Button("Print Rig Details - Dev")) { SaiDev.PrintAllBones(targetRig); }

            }
            else // no target rig selected
            {
                EditorGUILayout.Space();
                EditorGUILayout.HelpBox("Select a target to stream to.",MessageType.Info);
            }

        }

        private int DrawSourceList(SerializedProperty streamID)
        {
            _actors = new List<string>();
            if (_connection != null && _connection.Client != null)
            {
                var metadata = _connection.Client.MetaCollection;
                for (int i = 0; i < metadata.Count; i++) 
                {
                    var name = metadata[i].entityName;
                    if (string.IsNullOrEmpty(name))
                        continue;
                    _actors.Add(name);
                }
            }
            streamID.intValue = EditorGUILayout.Popup("Xsens Source", streamID.intValue, _actors.ToArray());
            return streamID.intValue;
        }

        private void DrawObjectIDList(int streamID, SerializedProperty objectID)
        {
            var objectIds = new List<string>();
            if (_connection != null && _connection.Client != null)
            {
                var scaleMetaCollection = _connection.Client.ScaleMetaCollection;
                if (scaleMetaCollection == null)
                    return;
                if (scaleMetaCollection.Count - 1 >= streamID)
                    objectIds = scaleMetaCollection[streamID].segmentNames;
            }
            objectID.intValue = EditorGUILayout.Popup("Object ID", objectID.intValue, objectIds.ToArray());
        }

        private void OpenRigMapping(GameObject targetRig) // deprecated
        {
            if (targetRig == null) return;
            Animator animator = targetRig.GetComponent<Animator>();
            if (animator == null || animator.avatar == null || !animator.avatar.isHuman) return;

            EditorApplication.delayCall += () =>
            {
                var avatarEditorType = typeof(UnityEditor.Editor).Assembly.GetType("UnityEditor.AvatarEditor");
                if (avatarEditorType == null) return;

                var avatarEditors = Resources.FindObjectsOfTypeAll(avatarEditorType);
                var avatarEditor = avatarEditors.Length > 0 ? avatarEditors[0] : null;
                if (avatarEditor)
                {
                    var configureAvatarMethod = avatarEditorType.GetMethod("SwitchToEditMode",
                        System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.NonPublic);
                    if (configureAvatarMethod == null) return;
                    configureAvatarMethod.Invoke(avatarEditor, null);
                    return;
                }
                Debug.Log("No Avatar Editor Found");
            };
        }

        // -------------------------------------------------------------------------------------------------------------
        // Controller
        ////Stream Settings ////
        private void OnConnectionsClicked()
        {
            GetWindow<Unity.LiveCapture.Editor.ConnectionsWindow>();
        }

        private void OnDiagnosticsClicked(){
            EditorWindow.GetWindow<XsensDiagnosticsWindow>();
        }
        private async void OnStartStreamClicked()
        {
            _isListening = !_isListening;
            if (_isListening)
            {
                var connectionType = typeof(XsensConnection);

                if (Unity.LiveCapture.ConnectionManager.Instance.TryGetConnection(out XsensConnection existingConnection))
                {
                    _connection = existingConnection;
                }
                else
                {
                    _connection = (XsensConnection)Unity.LiveCapture.ConnectionManager.Instance.CreateConnection(connectionType);
                }

                if (_connection)
                {
                    _connection.SetPort(_listenPort);
                    _connection.SetEnabled(true);
                }

                await Task.Run(SyncWithStatusRequest);
                await Task.Run(SyncWithInfoRequest);
            }
            else
            {
                if (_connection)
                {
                    _connection.SetEnabled(false);
                }
            }
        }
        /// / Sync with MVN Session ////
        private async Task SyncWithStatusRequest()
        {
            string response = await Task.Run(() => _mRemoteControl.SendMessage("status", _hostAddress, _remoteControlPort, getResponse: true));
            await Task.Delay(50);
            
            if (response == null || !response.StartsWith("<SessionStatusAck")) return;
            SessionStatusAck statuses = _mRemoteControl.ParseStatusMessage(response);
            _isLooping = statuses.IsRepeating;
            _isPlayback = statuses.IsPlaying;
            _isRecording = statuses.IsRecording;
            _timeline.CurrentFrame = statuses.CurrentFrame;
            _timeline.UpdatePlayheadPosition();
        }

        private async Task SyncWithInfoRequest()
        {
            string response = await Task.Run(() => _mRemoteControl.SendMessage("info", _hostAddress, _remoteControlPort, getResponse: true));
            await Task.Delay(50);
            
            if (response == null || !response.StartsWith("<SessionInfoAck")) return;
            SessionInfoAck info = _mRemoteControl.ParseInfoMessage(response);
            if (_timeline.TotalFrames != info.DurationFrames)
            {
                _timeline.RangeEndFrame = info.DurationFrames - 1;
                _timeline.RangeStartFrame = 0;
            }
            _timeline.TotalFrames = info.DurationFrames;
        }
        
        //// Playback Controls ////
        private async void OnTogglePlaybackClicked()
        {
            await Task.Run(SyncWithStatusRequest);
            await Task.Run(SyncWithInfoRequest);
            await Task.Run(() => _mRemoteControl.SendMessage("togglePlayback", _hostAddress, _remoteControlPort));
            await Task.Delay(50);
            await Task.Run(SyncWithStatusRequest);
        }

        private async void OnToggleLoopClicked()
        {
            await Task.Run(SyncWithStatusRequest);
            await Task.Run(SyncWithInfoRequest);
            await Task.Run(() => _mRemoteControl.SendMessage("toggleLoop", _hostAddress, _remoteControlPort));
            await Task.Delay(50);
            await Task.Run(SyncWithStatusRequest);
        }
        
        private async void HandleUserScrubPlayhead(int newFrame)
        {
            string request = $"<JumpToFrameReq frame=\"{_timeline.CurrentFrame}\"/>";
            await Task.Run(() => _mRemoteControl.SendMessage(request, _hostAddress, _remoteControlPort));
            await Task.Delay(50);
            await Task.Run(SyncWithStatusRequest);
            await Task.Run(SyncWithInfoRequest);
        }
        
        private async void SetPlayheadPosition(string request)
        {
            await Task.Run(() => _mRemoteControl.SendMessage(request, _hostAddress, _remoteControlPort));
            await Task.Delay(50);
            _isPlayback = false;
            await Task.Run(SyncWithStatusRequest);
            await Task.Run(SyncWithInfoRequest);
        }
        
        //// Live Session Controls ////
        private void OnStartRecordClicked()
        {
            _isRecording = !_isRecording;
                HandleMvnRecordSignal();
                HandleTakeRecorderSignal();

        }
            
            
        private void HandleMvnRecordSignal()
        {
            if (_isRecording)
            {
                var setNameReq =
                    $"<CaptureName> <Name VALUE=\"{_takeName}\" /> <Take VALUE=\"{_takeNumber}\" /> </CaptureName>";
                Task.Run(() =>
                    _mRemoteControl.SendMessage(setNameReq, _hostAddress, _remoteControlPort, getResponse: true));
                // combine the take name and number because mvn does not append take number automatically when receiving capture signal
                var takeTitle = $"{_takeName}-{_takeNumber:D3}"; 
                var captureStartReq =
                    $"<CaptureStart> <Name VALUE=\"{takeTitle}\" /> <Take VALUE=\"{_takeNumber}\" /> <TimeCode VALUE=\"\" /> <Notes></Notes> </CaptureStart>";
                Task.Run(() =>
                    _mRemoteControl.SendMessage(captureStartReq, _hostAddress, _remoteControlPort, getResponse: true));
            }
            else
            {
                var captureStopReq =
                    $"<CaptureStop> <Name VALUE=\"\" /> <TimeCode VALUE=\'\' /> <Notes></Notes> </CaptureStop>";
                Task.Run(() => _mRemoteControl.SendMessage(captureStopReq, _hostAddress, _remoteControlPort));
                _takeNumber += 1;
            }
        }
        
        private void HandleTakeRecorderSignal()
        {
            if (_isRecording)
            {
                
                if (_isTakeRecorderTriggerEnabled)
                {
                    if (Unity.LiveCapture.TakeRecorder.IsLive)
                    {
                        Unity.LiveCapture.TakeRecorder.StartRecording();
                    }
    
                    if (!Unity.LiveCapture.TakeRecorder.IsRecording())
                    {
                        Debug.LogError("Recording is not available. Check the take recorder status.");
                    }
                }
                    
            }
            
            else
            {
                if (Unity.LiveCapture.TakeRecorder.IsRecording())
                {
                    Unity.LiveCapture.TakeRecorder.StopRecording();
                    Debug.Log("Stopped recording with Take Recorder");
                    _takeNumber += 1;
                }
            }
        }
        private void OnActivateLiveSessionFeatureClicked(string request)
        {
            _mRemoteControl.SendMessage(request, _hostAddress, _remoteControlPort);
        }

        private void OnOpenTakeRecorderButtonClicked()
        {
            EditorApplication.ExecuteMenuItem("Window/Live Capture/Take Recorder");
        }
    }
}