using UnityEngine;
using UnityEditor;
using Unity.LiveCapture.Editor;
using UnityEngine.UIElements;

namespace Movella.Xsens.Editor
{
    
    /// <summary>
    /// Interface for setting the udp client address, port and opening and closing the connection.
    /// This is the widget that gets added to the Unity Live Capture Connections window.
    /// This widget is interfacing with the Live Capture API. This functionality could be moved to our main UI as
    /// Unity is ending its official support of the Live Capture Plugin. 
    /// </summary>
    [CustomEditor(typeof(XsensConnection), true)]
    public class XsensConnectionEditor : ConnectionEditor
    {
        static class Contents
        {
            public static GUIStyle enabledLabelStyle;
            public static GUIStyle disabledLabelStyle;
            public static readonly GUIContent DiagnosticsLabel = EditorGUIUtility.TrTextContent("Show Diagnostics", "Open the diagnostics panel for analyzing the Xsens data stream.");
            public static readonly GUIContent DiagnosticsButton = EditorGUIUtility.TrTextContent("Diagnostics", "Open the diagnostics panel for analyzing the Xsens data stream.");
            public static readonly GUIContent StreamPanelLabel = EditorGUIUtility.TrTextContent("Show Stream Panel", "Open the stream panel for managing the Xsens data stream.");
            public static readonly GUIContent StreamPanelButton = EditorGUIUtility.TrTextContent("Stream Panel", "Open the stream panel window for managing the Xsens data stream.");
            public static readonly GUILayoutOption[] ButtonOptions = { GUILayout.Width(160), GUILayout.Height(20) };
            public static readonly Color IndicatorConnectionPending = Color.yellow;
            
            public static void InitStyles()
            {
                if (enabledLabelStyle == null)
                {
                    enabledLabelStyle = new GUIStyle(EditorStyles.label);
                    enabledLabelStyle.normal.textColor = new Color(0.765f, 0.984f, 0.451f);
                }
        
                if (disabledLabelStyle == null)
                {
                    disabledLabelStyle = new GUIStyle(EditorStyles.label);
                    disabledLabelStyle.normal.textColor = Color.gray;
                }
            }
        }

        SerializedProperty m_StartOnPlayProperty;
        SerializedProperty m_PortProperty;

        XsensConnection m_Connection;

        protected override void OnEnable()
        {
            base.OnEnable();

            m_StartOnPlayProperty = serializedObject.FindProperty(XsensConnection.Names.StartOnPlay);
            m_PortProperty = serializedObject.FindProperty(XsensConnection.Names.Port);

            m_Connection = target as XsensConnection;
        }

        protected override VisualElement CreateInfoGUI()
        {
            GetToolbar().Indicator.schedule.Execute(UpdateConnectionIndicator).Every(100);
            
            return base.CreateInfoGUI();
        }

        protected override VisualElement CreateSettingsGUI()
        {
            return CreateIMGUIContainer(DrawSettingsGUI);
        }

        void DrawSettingsGUI()
        {
            serializedObject.Update();

            EditorGUILayout.PropertyField(m_StartOnPlayProperty);
            DrawPortGUI();
            DrawStreamEntityToggles();
            DrawDiagnosticsButton();
            DrawXsensLiveStreamButton();

            serializedObject.ApplyModifiedProperties();
        }

        void DrawPortGUI()
        {
            using (var check = new EditorGUI.ChangeCheckScope())
            {
                EditorGUILayout.PropertyField(m_PortProperty);

                if (check.changed)
                    m_Connection.Stop();
            }
        }

        void DrawStreamEntityToggles()
        {
          if (m_Connection.Client == null) return;
          EditorGUILayout.LabelField("Stream Entities", EditorStyles.boldLabel);
          for (int i = 0; i < m_Connection.Client.m_Frames.Length; i++)
          {
              var frame = m_Connection.Client.m_Frames[i];
              DrawStreamEntityToggle(frame, i);
          }
        }

        void DrawStreamEntityToggle(FrameData frame, int streamIndex)
        {
            using (new EditorGUILayout.HorizontalScope())
            {
                var isEnabled = m_Connection.Client.StatusCollection[streamIndex];

                Contents.InitStyles();
                bool newEnabled = EditorGUILayout.Toggle(GUIContent.none, isEnabled, GUILayout.Width(20));
                var streamLabel = string.IsNullOrEmpty(frame.metadata.entityName)
                    ? $"{streamIndex + 1}"
                    : $"{streamIndex + 1}: {frame.metadata.entityName}";
                EditorGUILayout.LabelField(streamLabel, isEnabled ? Contents.enabledLabelStyle : Contents.disabledLabelStyle);

                if (newEnabled == isEnabled || frame.enabled == newEnabled) return;
                
                frame.enabled = newEnabled;
                m_Connection.Client.StatusCollection[streamIndex] = newEnabled;
                EditorUtility.SetDirty(target);
            }
        }

        void DrawDiagnosticsButton()
        {
            using (new GUILayout.HorizontalScope())
            {
                EditorGUILayout.PrefixLabel(Contents.DiagnosticsLabel);

                if (GUILayout.Button(Contents.DiagnosticsButton, Contents.ButtonOptions))
                    EditorWindow.GetWindow<XsensDiagnosticsWindow>().Show();
            }
        }
        
        void DrawXsensLiveStreamButton()
        {
            using (new GUILayout.HorizontalScope())
            {
                EditorGUILayout.PrefixLabel(Contents.StreamPanelLabel);

                if (GUILayout.Button(Contents.StreamPanelButton, Contents.ButtonOptions))
                    EditorWindow.GetWindow<XsensLiveStreamEditor>().Show();
            }
        }

        protected override void OnConnectionChanged()
        {
            base.OnConnectionChanged();

            UpdateConnectionIndicator(); 
        }

        void UpdateConnectionIndicator()
        {
            var indicator = GetToolbar().Indicator;

            indicator.style.backgroundColor = m_Connection.IsEnabled() && !m_Connection.IsConnected ? 
                Contents.IndicatorConnectionPending : 
                StyleKeyword.Null;
        }
    }
}
