﻿using System;
using UnityEditor;
using UnityEngine;
using UnityEditorInternal;
using System.Collections.Generic;
using System.Linq;
using Codice.Client.BaseCommands.BranchExplorer;
using Object = UnityEngine.Object;

[System.Serializable]
// region Target Element classes
public class StreamTarget
{
    public bool enable = true;

    public string name;
    public string type = "Actor"; // Actor or Object

    public string[] streamChannels = Array.Empty<string>();
    public int selectedStreamChannelIndex = 0;

    public GameObject targetRig;

    public string[] propChannels = Array.Empty<string>();
    public int selectedPropChannelIndex = 0;
    public GameObject propChild;
    public GameObject propParent;
    public int selectedPropParentIndex = 0;
}


public class StreamTargets : ScriptableObject
{
    public List<StreamTarget> streamTargets = new List<StreamTarget>();
}
// endregion

namespace Movella.Xsens.Editor
{
    public class StreamTargetsListUI : ReorderableList
    {
        private SerializedProperty _listElements;

        public StreamTargetsListUI(SerializedObject streamTargetsSerializedObject, SerializedProperty listElements,
            List<string> streamTargets)
            : base(streamTargetsSerializedObject, listElements, true, true, true, true)
        {
            _listElements = listElements;
            elementHeight = EditorGUIUtility.singleLineHeight * 3 + 8;

            // ---------------------------------------------------------------------------------------------------------
            // region Draw UI Callbacks
            drawHeaderCallback = rect => EditorGUI.LabelField(rect, "Target Characters");

            drawElementCallback = (rect, elementIndex, isActive, isFocused) =>
            {
                // Get the properties of the current element
                var element = _listElements.GetArrayElementAtIndex(elementIndex);
                var enableProperty = element.FindPropertyRelative("enable");
                var streamChannelsProperty = element.FindPropertyRelative("streamChannels");
                var selectedStreamChannelIndexProperty = element.FindPropertyRelative("selectedStreamChannelIndex");
                var targetRigProperty = element.FindPropertyRelative("targetRig");
                var propChannelsProperty = element.FindPropertyRelative("propChannels");
                var selectedPropChannelIndexProperty = element.FindPropertyRelative("selectedPropChannelIndex");
                var propChildProperty = element.FindPropertyRelative("propChild");
                var propParentProperty = element.FindPropertyRelative("propParent");
                var selectedPropParentIndexProperty = element.FindPropertyRelative("selectedPropParentIndex");

                // Row 1
                rect.y += 2;
                Dictionary<string, float> elementWidths = new Dictionary<string, float>();
                elementWidths.Add("enable", 20f);
                elementWidths.Add("label", 100f);
                elementWidths.Add("streamChannel", 100f);
                elementWidths.Add("targetRig", rect.width - elementWidths["enable"] - elementWidths["label"] - elementWidths["streamChannel"]);
                
                DrawToggle(rect, enableProperty);
                GUI.color = enableProperty.boolValue ? Color.green : Color.white;
                DrawElementLabel(targetRigProperty, rect, elementWidths);
                GUI.color = Color.white;
                DrawStreamChannelDropdown(streamChannelsProperty, selectedStreamChannelIndexProperty, rect, elementWidths);
                DrawTargetRigField(targetRigProperty, rect, elementWidths);
                
                // Row 2 (labels drawn above the fields)
                rect.y += EditorGUIUtility.singleLineHeight + 2;
                elementWidths.Clear();
                elementWidths.Add("tposeButton", 80f);
                elementWidths.Add("openMappingButton", 80f);
                elementWidths.Add("propChannel", 100f);
                elementWidths.Add("propChild", 150f);
                elementWidths.Add("propParent", rect.width - elementWidths["tposeButton"] - elementWidths["propChannel"] - elementWidths["propChild"]);
                
                DrawTposeButton(rect, elementWidths);
                
                DrawOpenMappingButton(rect, elementWidths, targetRigProperty);
                
                DrawPropChannelLabel(rect, elementWidths);
                DrawPropChannelDropdown(rect, elementWidths, propChannelsProperty, selectedPropChannelIndexProperty);
                DrawPropChildLabel(rect, elementWidths);
                DrawPropChildField(propChildProperty, rect, elementWidths);
                DrawPropParentLabel(rect, elementWidths, propParentProperty);
                DrawPropParentDropdown(rect, elementWidths, propParentProperty, targetRigProperty, selectedPropParentIndexProperty);
                
                _listElements.serializedObject.ApplyModifiedProperties();
            };

            onAddCallback = (reorderableList) =>
            {
                // Add the selected prefab to the stream targets list else add an empty element
                if (Selection.gameObjects != null && Selection.gameObjects.Length > 0)
                {
                    var itemAdded = false;
                    foreach (var gameObject in Selection.gameObjects)
                    {
                        if (!PrefabUtility.IsAnyPrefabInstanceRoot(gameObject) || streamTargets.Contains(gameObject.name)) continue;
                        streamTargets.Add(gameObject.name);
                        AddStreamTarget(gameObject);
                        itemAdded = true;
                    }
                    if (!itemAdded) AddStreamTarget();
                }
                else AddStreamTarget();
            };
        }
        //endregion
        // -------------------------------------------------------------------------------------------------------------
        // region Draw UI methods
        private void DrawToggle(Rect rect, SerializedProperty enableProperty)
        {
            float toggleWidth = 20f;
            var enabledToggleLayout = new Rect(rect.x, rect.y, toggleWidth, EditorGUIUtility.singleLineHeight);
            enableProperty.boolValue = EditorGUI.Toggle(enabledToggleLayout, enableProperty.boolValue);
        }

        private void DrawElementLabel(SerializedProperty targetRigProperty, Rect rect, Dictionary<string, float> elementWidths)
        {
            GUIContent labelContent = new GUIContent(targetRigProperty.objectReferenceValue != null
                ? targetRigProperty.objectReferenceValue.name : "None");
            var elementLabelLayout = new Rect(rect.x + elementWidths["enable"], rect.y, elementWidths["streamChannel"], EditorGUIUtility.singleLineHeight);
            EditorGUI.LabelField(elementLabelLayout, labelContent);
        }

        private void DrawStreamChannelDropdown(SerializedProperty streamChannelsProperty, SerializedProperty selectedIndexProperty, Rect rect, Dictionary<string, float> elementWidths)
        {
            string[] streamChannelNames = new string[streamChannelsProperty.arraySize];
            for (int i = 0; i < streamChannelsProperty.arraySize; i++)
            {
                streamChannelNames[i] = streamChannelsProperty.GetArrayElementAtIndex(i).stringValue;
            }

            var streamChannelLayout = new Rect(rect.x + elementWidths["enable"] + elementWidths["label"], rect.y, elementWidths["streamChannel"],
                EditorGUIUtility.singleLineHeight);
            selectedIndexProperty.intValue = EditorGUI.Popup(streamChannelLayout, selectedIndexProperty.intValue, streamChannelNames);
            streamChannelsProperty.GetArrayElementAtIndex(selectedIndexProperty.intValue).stringValue = streamChannelNames[selectedIndexProperty.intValue];
        }
        
        private void DrawTargetRigField(SerializedProperty targetRigProperty, Rect rect, Dictionary<string, float> elementWidths)
        {
            var targetRigLayout = new Rect(rect.x + elementWidths["enable"] + elementWidths["label"] + elementWidths["streamChannel"], rect.y, rect.width - elementWidths["enable"] - elementWidths["label"] - elementWidths["streamChannel"],
                EditorGUIUtility.singleLineHeight);
            targetRigProperty.objectReferenceValue = EditorGUI.ObjectField(targetRigLayout, targetRigProperty.objectReferenceValue, typeof(GameObject), true);
        }

        private void DrawTposeButton(Rect rect, Dictionary<string, float> elementWidths)
        {
            var tPoseButtonContent = new GUIContent("Reset Pose", "Restore the default pose of the selected target rig.");
            var tposeButtonLayout = new Rect(rect.x, rect.y + 20, elementWidths["tposeButton"], EditorGUIUtility.singleLineHeight);
            if (GUI.Button(tposeButtonLayout, tPoseButtonContent)) RestoreTpose();
        }
        
        private void DrawOpenMappingButton(Rect rect, Dictionary<string, float> elementWidths, SerializedProperty targetRigProperty)
        {
            var openMappingButtonContent = new GUIContent("Rig Mapping", "Open the mapping window for the selected target rig.");
            var openMappingButtonLayout = new Rect(rect.x, rect.y, elementWidths["openMappingButton"], EditorGUIUtility.singleLineHeight);
            if (GUI.Button(openMappingButtonLayout, openMappingButtonContent)) OpenRigMapping(targetRigProperty);
        }
        
        private void DrawPropChannelLabel(Rect rect, Dictionary<string, float> elementWidths)
        {
            var propChannelLabelLayout = new Rect(rect.x + elementWidths["tposeButton"], rect.y, elementWidths["propChannel"], EditorGUIUtility.singleLineHeight);
            EditorGUI.LabelField(propChannelLabelLayout, "Prop Channel");
        }
        
        private void DrawPropChannelDropdown(Rect rect, Dictionary<string, float> elementWidths, SerializedProperty propChannelsProperty, SerializedProperty selectedPropChannelIndexProperty)
        {
            var propChannelLayout = new Rect(rect.x + elementWidths["tposeButton"], rect.y + EditorGUIUtility.singleLineHeight, elementWidths["propChannel"], EditorGUIUtility.singleLineHeight);
            string[] propChannelNames = new string[propChannelsProperty.arraySize];
            for (int i = 0; i < propChannelsProperty.arraySize; i++)
            {
                propChannelNames[i] = propChannelsProperty.GetArrayElementAtIndex(i).stringValue;
            }
            selectedPropChannelIndexProperty.intValue = EditorGUI.Popup(propChannelLayout, selectedPropChannelIndexProperty.intValue, propChannelNames);
            propChannelsProperty.GetArrayElementAtIndex(selectedPropChannelIndexProperty.intValue).stringValue = propChannelNames[selectedPropChannelIndexProperty.intValue];
        }
        
        private void DrawPropChildLabel(Rect rect, Dictionary<string, float> elementWidths)
        {
            var propChildLabelLayout = new Rect(rect.x + elementWidths["tposeButton"] + elementWidths["propChannel"], rect.y, elementWidths["propChild"], EditorGUIUtility.singleLineHeight);
            EditorGUI.LabelField(propChildLabelLayout, "Prop Child");
        }
        
        private void DrawPropChildField(SerializedProperty propChildProperty, Rect rect, Dictionary<string, float> elementWidths)
        {
            var propChildLayout = new Rect(rect.x + elementWidths["tposeButton"] + elementWidths["propChannel"], rect.y + EditorGUIUtility.singleLineHeight, elementWidths["propChild"], EditorGUIUtility.singleLineHeight);
            propChildProperty.objectReferenceValue = EditorGUI.ObjectField(propChildLayout, propChildProperty.objectReferenceValue, typeof(GameObject), true);
        }
        
        private void DrawPropParentLabel(Rect rect, Dictionary<string, float> elementWidths, SerializedProperty propParentProperty)
        {
            var propParentLabelLayout = new Rect(rect.x + elementWidths["tposeButton"] + elementWidths["propChannel"] + elementWidths["propChild"], rect.y, elementWidths["propParent"], EditorGUIUtility.singleLineHeight);
            EditorGUI.LabelField(propParentLabelLayout, "Prop Parent");
        }

        private void DrawPropParentDropdown(Rect rect, Dictionary<string, float> elementWidths, SerializedProperty propParentProperty, SerializedProperty targetRigProperty, SerializedProperty selectedPropParentIndexProperty)
        {
            if (targetRigProperty.objectReferenceValue == null)
            {
                EditorGUI.LabelField(new Rect(rect.x + elementWidths["tposeButton"] + elementWidths["propChannel"] + elementWidths["propChild"], rect.y + EditorGUIUtility.singleLineHeight, elementWidths["propParent"], EditorGUIUtility.singleLineHeight), "Select a target rig.");
                return;
            }

            var propParentLayout = new Rect(rect.x + elementWidths["tposeButton"] + elementWidths["propChannel"] + elementWidths["propChild"], rect.y + EditorGUIUtility.singleLineHeight, elementWidths["propParent"], EditorGUIUtility.singleLineHeight);
            var boneNames = new List<string>();
            var selectedRig = targetRigProperty.objectReferenceValue;
            if (selectedRig is GameObject selectedGameObject)
            {
                var animator = selectedGameObject.GetComponent<Animator>();
                if (animator != null && animator.avatar != null && animator.avatar.isHuman)
                {
                    boneNames.AddRange(
                        from HumanBodyBones bone in Enum.GetValues(typeof(HumanBodyBones))
                        where bone != HumanBodyBones.LastBone
                        let boneTransform = animator.GetBoneTransform(bone)
                        where boneTransform != null
                        select bone.ToString()
                    );
                }
            }
            selectedPropParentIndexProperty.intValue = EditorGUI.Popup(propParentLayout, selectedPropParentIndexProperty.intValue, boneNames.ToArray());
        }
        // endregion                
        // -------------------------------------------------------------------------------------------------------------
        // region Element callbacks
        private void AddStreamTarget(GameObject gameObject = null)
        {
            _listElements.arraySize++;
            var newElement = _listElements.GetArrayElementAtIndex(_listElements.arraySize - 1);

            var enableProperty = newElement.FindPropertyRelative("enable");
            enableProperty.boolValue = gameObject;

            // FUTURE: This will be replaced with the actual stream channel names from the xsens stream
            var streamChannelsProperty = newElement.FindPropertyRelative("streamChannels");
            streamChannelsProperty.arraySize = 5;
            for (int i = 0; i < streamChannelsProperty.arraySize; i++)
            {
                streamChannelsProperty.GetArrayElementAtIndex(i).stringValue = "Stream_" + (i + 1);
            }

            var selectedChannelIndexProperty = newElement.FindPropertyRelative("selectedStreamChannelIndex");
            selectedChannelIndexProperty.intValue = 0;

            var targetRig = newElement.FindPropertyRelative("targetRig");
            targetRig.objectReferenceValue = gameObject;

            // FUTURE: This will be replaced with the actual prop segments from the xsens stream and will be empty if there is no prop in the character stream
            var propChannelsProperty = newElement.FindPropertyRelative("propChannels");
            propChannelsProperty.arraySize = 4;
            for (int i = 0; i < propChannelsProperty.arraySize; i++)
            {
                propChannelsProperty.GetArrayElementAtIndex(i).stringValue = (i + 1).ToString();
            }

            var propChild = newElement.FindPropertyRelative("propChild");
            propChild.objectReferenceValue = null;

            var propParent = newElement.FindPropertyRelative("propParent");
            propParent.objectReferenceValue = null;
            
            var selectedPropParentIndex = newElement.FindPropertyRelative("selectedPropParentIndex");
            selectedPropParentIndex.intValue = 0;

            _listElements.serializedObject.ApplyModifiedProperties();
        }
        
        private void RestoreTpose()
        {
            Debug.Log("Restore Tpose Button clicked!");
        }
        
        private void OpenRigMapping(SerializedProperty targetRigProperty)
        {
            _listElements.serializedObject.ApplyModifiedProperties();

            if (targetRigProperty.objectReferenceValue is not GameObject targetRig ||
                targetRig.GetComponent<Animator>()?.avatar is not { isHuman: true } avatar) return;
            
            EditorApplication.delayCall += () =>
            {
                var avatarEditorType = typeof(UnityEditor.Editor).Assembly.GetType("UnityEditor.AvatarEditor");
                var avatarEditor = Resources.FindObjectsOfTypeAll(avatarEditorType).FirstOrDefault();
                if (avatarEditor != null)
                {
                    var configureAvatarMethod = avatarEditorType.GetMethod("SwitchToEditMode",
                        System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.NonPublic);
                    configureAvatarMethod?.Invoke(avatarEditor, null);
                }
            };
        }
        
        public void DrawCachedList(List<XsensLiveStreamEditor.StreamTargetState> lastKnownState)
        {
            if (lastKnownState != null && lastKnownState.Count > 0)
            {
                // Logic to draw the cached list using lastKnownState
                foreach (var listElement in lastKnownState)
                {
                    // use the stored rig reference to re-create the element
                    var targetRig = listElement.TargetRig;
                    if (targetRig == null) continue;
                    Debug.Log("Creating element for " + targetRig.name + " From cached list");
                    AddStreamTarget(targetRig);
                    // get the new element and set the properties
                    var newElement = _listElements.GetArrayElementAtIndex(_listElements.arraySize - 1);
                    // TODO: Set the properties of the new element
                }
            }
        }
    }
}