﻿using UnityEngine;
using UnityEditor;

namespace Movella.Xsens.Editor
{
    public class PlaybackTimeline
    {
        private float _playheadPosition;
        private bool _isDragging;
        private const int StartFrame = 0;
        public int RangeStartFrame = 500;
        public int CurrentFrame = 1000;
        public int RangeEndFrame = 1500;
        public int TotalFrames = 3000;
        private int _finalFrameNum = 2999;

        public delegate void FrameUpdatedEventHandler(int newFrame);
        public event FrameUpdatedEventHandler OnFrameUpdated;

        public void OnEnable()
        {
            // Initialize properties here
        }

        public void OnGUI()
        {
            GUILayout.BeginHorizontal();
            GUILayout.Label(StartFrame.ToString(), GUILayout.Width(20));
            RangeStartFrame = EditorGUILayout.IntField("", RangeStartFrame, GUILayout.Width(40));
            Rect timelineRect = GUILayoutUtility.GetRect(0, 20, GUILayout.ExpandWidth(true));
            DrawTimeline(timelineRect);
            RangeEndFrame = EditorGUILayout.IntField("", RangeEndFrame, GUILayout.Width(40));
            _finalFrameNum = _finalFrameNum = TotalFrames - 1;
            GUILayout.Label(_finalFrameNum.ToString(), GUILayout.Width(40));
            GUILayout.EndHorizontal();
        }

        private void DrawTimeline(Rect rect)
        {
            EditorGUI.DrawRect(rect, Color.clear);
            float borderThickness = 1f;
            
            EditorGUI.DrawRect(new Rect(rect.x, rect.y, rect.width, borderThickness), Color.black); // Top border
            EditorGUI.DrawRect(new Rect(rect.x, rect.yMax - borderThickness, rect.width, borderThickness), Color.black); // Bottom border
            EditorGUI.DrawRect(new Rect(rect.x, rect.y, borderThickness, rect.height), Color.black); // Left border
            EditorGUI.DrawRect(new Rect(rect.xMax - borderThickness, rect.y, borderThickness, rect.height), Color.black); // Right border

            float playheadX = Mathf.Lerp(rect.x, rect.xMax, (_playheadPosition - RangeStartFrame) / (RangeEndFrame - RangeStartFrame));
            Rect playheadRect = new Rect(playheadX - 1, rect.y, 5, rect.height);
            EditorGUI.DrawRect(playheadRect, new Color(0.6f, 0.851f, 0.918f));

            DetermineCurrentFrameLabelPosition(playheadX, rect);
            HandleUserScrubEvents(rect, playheadRect);

        }
        
        private void DetermineCurrentFrameLabelPosition(float playheadX, Rect timelineRect)
        {
            float labelX = playheadX + 5;
            if (labelX + 40 > timelineRect.xMax)
            {
                labelX = playheadX - 40;
            }
            GUI.Label(new Rect(labelX, timelineRect.y, 50, 20), CurrentFrame.ToString());
        }

        private void HandleUserScrubEvents(Rect timelineRect, Rect playheadRect)
        {
            var e = Event.current;
            switch (e.type)
            {
                case EventType.MouseDown when playheadRect.Contains(e.mousePosition):
                    _isDragging = true;
                    e.Use();
                    break;
                case EventType.MouseDrag when _isDragging:
                    _playheadPosition = Mathf.Clamp((e.mousePosition.x - timelineRect.x) / timelineRect.width * (RangeEndFrame - RangeStartFrame) + RangeStartFrame, RangeStartFrame, RangeEndFrame);
                    CurrentFrame = Mathf.FloorToInt(_playheadPosition);
                    OnFrameUpdated?.Invoke(CurrentFrame);
                    e.Use();
                    break;
                case EventType.MouseUp when _isDragging:
                    _isDragging = false;
                    e.Use();
                    break;
            }
        }

        public void UpdatePlayheadPosition()
        {
            _playheadPosition = Mathf.Clamp(CurrentFrame, RangeStartFrame, RangeEndFrame);
        }
    }
}