﻿using System;
using UnityEditor;
using UnityEngine;
using UnityEditorInternal;
using System.Collections.Generic;
using System.Linq;

[System.Serializable]
// region Target Element classes
public class ObjectStreamTarget
{
    public bool enable = true;

    public string[] streamChannels = Array.Empty<string>();
    public int selectedStreamChannelIndex = 0;
    public GameObject targetObject;
}

public class ObjectStreamTargets : ScriptableObject
{
    public List<ObjectStreamTarget> objectStreamTargets = new List<ObjectStreamTarget>();
}


namespace Movella.Xsens.Editor
{
    public class ObjectStreamTargetsListUI : ReorderableList
    {
        private SerializedProperty _listElements;

        public ObjectStreamTargetsListUI(SerializedObject objectStreamTargetsSerializedObject,
            SerializedProperty listElements,
            List<string> objectStreamTargets)
            : base(objectStreamTargetsSerializedObject, listElements, true, true, true, true)
        {
            _listElements = listElements;
            elementHeight = EditorGUIUtility.singleLineHeight * 3 + 8;

            // ---------------------------------------------------------------------------------------------------------
            // region Draw UI Callbacks
            drawHeaderCallback = rect => EditorGUI.LabelField(rect, "Target Objects");

            drawElementCallback = (rect, elementIndex, isActive, isFocused) =>
            {
                // Get the properties of the current element
                var element = _listElements.GetArrayElementAtIndex(elementIndex);
                var enableProperty = element.FindPropertyRelative("enable");
                var streamChannelsProperty = element.FindPropertyRelative("streamChannels");
                var selectedStreamChannelIndexProperty = element.FindPropertyRelative("selectedStreamChannelIndex");

                // Row 1
                rect.y += 2;
                Dictionary<string, float> elementWidths = new Dictionary<string, float>();
                elementWidths.Add("enable", 20f);
                elementWidths.Add("label", 100f);
                elementWidths.Add("streamChannel", 100f);
                
                DrawToggle(rect, enableProperty);
                DrawStreamChannelDropdown(streamChannelsProperty, selectedStreamChannelIndexProperty, rect, elementWidths);
                _listElements.serializedObject.ApplyModifiedProperties();
            };

            onAddCallback = (reorderableList) =>
            {
                // Add the selected prefab to the stream targets list else add an empty element
                if (Selection.gameObjects != null && Selection.gameObjects.Length > 0)
                {
                    var itemAdded = false;
                    foreach (var gameObject in Selection.gameObjects)
                    {
                        if (!PrefabUtility.IsAnyPrefabInstanceRoot(gameObject) || objectStreamTargets.Contains(gameObject.name)) continue;
                        objectStreamTargets.Add(gameObject.name);
                        AddStreamTarget(gameObject);
                        itemAdded = true;
                    }
                    if (!itemAdded) AddStreamTarget();
                }
                else AddStreamTarget();
            };
        }
        //endregion
        // -------------------------------------------------------------------------------------------------------------
        // region Draw UI methods
        private void DrawToggle(Rect rect, SerializedProperty enableProperty)
        {
            float toggleWidth = 20f;
            var enabledToggleLayout = new Rect(rect.x, rect.y, toggleWidth, EditorGUIUtility.singleLineHeight);
            enableProperty.boolValue = EditorGUI.Toggle(enabledToggleLayout, enableProperty.boolValue);
        }

        private void DrawElementLabel(SerializedProperty targetRigProperty, Rect rect, Dictionary<string, float> elementWidths)
        {
            GUIContent labelContent = new GUIContent(targetRigProperty.objectReferenceValue != null
                ? targetRigProperty.objectReferenceValue.name : "None");
            var elementLabelLayout = new Rect(rect.x + elementWidths["enable"], rect.y, elementWidths["streamChannel"], EditorGUIUtility.singleLineHeight);
            EditorGUI.LabelField(elementLabelLayout, labelContent);
        }

        private void DrawStreamChannelDropdown(SerializedProperty streamChannelsProperty, SerializedProperty selectedIndexProperty, Rect rect, Dictionary<string, float> elementWidths)
        {
            string[] streamChannelNames = new string[streamChannelsProperty.arraySize];
            for (int i = 0; i < streamChannelsProperty.arraySize; i++)
            {
                streamChannelNames[i] = streamChannelsProperty.GetArrayElementAtIndex(i).stringValue;
            }

            var streamChannelLayout = new Rect(rect.x + elementWidths["enable"] + elementWidths["label"], rect.y, elementWidths["streamChannel"],
                EditorGUIUtility.singleLineHeight);
            selectedIndexProperty.intValue = EditorGUI.Popup(streamChannelLayout, selectedIndexProperty.intValue, streamChannelNames);
            streamChannelsProperty.GetArrayElementAtIndex(selectedIndexProperty.intValue).stringValue = streamChannelNames[selectedIndexProperty.intValue];
        }
        
        // endregion                
        // -------------------------------------------------------------------------------------------------------------
        // region Element callbacks
        private void AddStreamTarget(GameObject gameObject = null)
        {
            _listElements.arraySize++;
            var newElement = _listElements.GetArrayElementAtIndex(_listElements.arraySize - 1);

            var enableProperty = newElement.FindPropertyRelative("enable");
            enableProperty.boolValue = gameObject;

            // FUTURE: This will be replaced with the actual stream channel names from the xsens stream
            var streamChannelsProperty = newElement.FindPropertyRelative("streamChannels");
            streamChannelsProperty.arraySize = 5;
            for (int i = 0; i < streamChannelsProperty.arraySize; i++)
            {
                streamChannelsProperty.GetArrayElementAtIndex(i).stringValue = "Stream_" + (i + 1);
            }

            var selectedChannelIndexProperty = newElement.FindPropertyRelative("selectedStreamChannelIndex");
            selectedChannelIndexProperty.intValue = 0;

            var targetRig = newElement.FindPropertyRelative("targetObject");
            targetRig.objectReferenceValue = gameObject;

            _listElements.serializedObject.ApplyModifiedProperties();
        }
        
        
        public void DrawCachedList(List<XsensLiveStreamEditor.StreamTargetState> lastKnownState)
        {
            if (lastKnownState != null && lastKnownState.Count > 0)
            {
                // Logic to draw the cached list using lastKnownState
                foreach (var listElement in lastKnownState)
                {
                    // use the stored rig reference to re-create the element
                    var targetObject = listElement.TargetRig;
                    if (targetObject == null) continue;
                    Debug.Log("Creating element for " + targetObject.name + " From cached list");
                    AddStreamTarget(targetObject);
                    // get the new element and set the properties
                    var newElement = _listElements.GetArrayElementAtIndex(_listElements.arraySize - 1);
                    // TODO: Set the properties of the new element
                }
            }
        }
    }
}