# *********************************************************************************************************************
# segment_maps.py provides mappings and definitions for motion capture data processing:
# Maps:
#   segment_map: Mapping system for converting Xsens segment IDs to human-readable segment names
#   stream_hierarchy_map: Definition of the hierarchical relationships between body segments
#   bone_orientations_map: Definition of the bone orientations for each segment
#   expected_prop_names: list of expected prop names from MVN Animate.
#   possible_prop_segments: A list of segments supporting prop connection.
#   finger_joint_offsets: Translation offsets from parent joint for fingers
#   common_remapings, auto_remapings: Dictionaries for common remappings to translate between different naming conventions.
# *********************************************************************************************************************
from mathutils import Vector

# ===========================================================================================
segment_map = {
    1: "Pelvis",
    2: "L5",
    3: "L3",
    4: "T12",
    5: "T8",
    6: "Neck",
    7: "Head",
    8: "RightShoulder",
    9: "RightUpperArm",
    10: "RightForeArm",
    11: "RightHand",
    12: "LeftShoulder",
    13: "LeftUpperArm",
    14: "LeftForeArm",
    15: "LeftHand",
    16: "RightUpperLeg",
    17: "RightLowerLeg",
    18: "RightFoot",
    19: "RightToe",
    20: "LeftUpperLeg",
    21: "LeftLowerLeg",
    22: "LeftFoot",
    23: "LeftToe",
    24: "Prop1",
    25: "Prop2",
    26: "Prop3",
    27: "Prop4",
    # --- Assuming 4 props (24-27), the Carpus of the left hand starts at 28 ---
    28: "LeftCarpus",
    29: "LeftFirstMC",
    30: "LeftFirstPP",
    31: "LeftFirstDP",
    32: "LeftSecondMC",
    33: "LeftSecondPP",
    34: "LeftSecondMP",
    35: "LeftSecondDP",
    36: "LeftThirdMC",
    37: "LeftThirdPP",
    38: "LeftThirdMP",
    39: "LeftThirdDP",
    40: "LeftFourthMC",
    41: "LeftFourthPP",
    42: "LeftFourthMP",
    43: "LeftFourthDP",
    44: "LeftFifthMC",
    45: "LeftFifthPP",
    46: "LeftFifthMP",
    47: "LeftFifthDP",
    48: "RightCarpus",
    49: "RightFirstMC",
    50: "RightFirstPP",
    51: "RightFirstDP",
    52: "RightSecondMC",
    53: "RightSecondPP",
    54: "RightSecondMP",
    55: "RightSecondDP",
    56: "RightThirdMC",
    57: "RightThirdPP",
    58: "RightThirdMP",
    59: "RightThirdDP",
    60: "RightFourthMC",
    61: "RightFourthPP",
    62: "RightFourthMP",
    63: "RightFourthDP",
    64: "RightFifthMC",
    65: "RightFifthPP",
    66: "RightFifthMP",
    67: "RightFifthDP",
}

stream_hierarchy_map = [
    ("Pelvis", None),
    ("L5", "Pelvis"),
    ("L3", "L5"),
    ("T12", "L3"),
    ("T8", "T12"),
    ("Neck", "T8"),
    ("Head", "Neck"),
    # --- Right arm ---
    ("RightShoulder", "T8"),
    ("RightUpperArm", "RightShoulder"),
    ("RightForeArm", "RightUpperArm"),
    ("RightHand", "RightForeArm"),
    ("RightCarpus", "RightHand"),
    # --- Right hand fingers ---
    ("RightFirstMC", "RightCarpus"),
    ("RightFirstPP", "RightFirstMC"),
    ("RightFirstDP", "RightFirstPP"),
    ("RightSecondMC", "RightCarpus"),
    ("RightSecondPP", "RightSecondMC"),
    ("RightSecondMP", "RightSecondPP"),
    ("RightSecondDP", "RightSecondMP"),
    ("RightThirdMC", "RightCarpus"),
    ("RightThirdPP", "RightThirdMC"),
    ("RightThirdMP", "RightThirdPP"),
    ("RightThirdDP", "RightThirdMP"),
    ("RightFourthMC", "RightCarpus"),
    ("RightFourthPP", "RightFourthMC"),
    ("RightFourthMP", "RightFourthPP"),
    ("RightFourthDP", "RightFourthMP"),
    ("RightFifthMC", "RightCarpus"),
    ("RightFifthPP", "RightFifthMC"),
    ("RightFifthMP", "RightFifthPP"),
    ("RightFifthDP", "RightFifthMP"),
    # --- Left arm ---
    ("LeftShoulder", "T8"),
    ("LeftUpperArm", "LeftShoulder"),
    ("LeftForeArm", "LeftUpperArm"),
    ("LeftHand", "LeftForeArm"),
    ("LeftCarpus", "LeftHand"),
    # --- Left hand fingers ---
    ("LeftFirstMC", "LeftCarpus"),
    ("LeftFirstPP", "LeftFirstMC"),
    ("LeftFirstDP", "LeftFirstPP"),
    ("LeftSecondMC", "LeftCarpus"),
    ("LeftSecondPP", "LeftSecondMC"),
    ("LeftSecondMP", "LeftSecondPP"),
    ("LeftSecondDP", "LeftSecondMP"),
    ("LeftThirdMC", "LeftCarpus"),
    ("LeftThirdPP", "LeftThirdMC"),
    ("LeftThirdMP", "LeftThirdPP"),
    ("LeftThirdDP", "LeftThirdMP"),
    ("LeftFourthMC", "LeftCarpus"),
    ("LeftFourthPP", "LeftFourthMC"),
    ("LeftFourthMP", "LeftFourthPP"),
    ("LeftFourthDP", "LeftFourthMP"),
    ("LeftFifthMC", "LeftCarpus"),
    ("LeftFifthPP", "LeftFifthMC"),
    ("LeftFifthMP", "LeftFifthPP"),
    ("LeftFifthDP", "LeftFifthMP"),
    # --- Right leg ---
    ("RightUpperLeg", "Pelvis"),
    ("RightLowerLeg", "RightUpperLeg"),
    ("RightFoot", "RightLowerLeg"),
    ("RightToe", "RightFoot"),
    # --- Left leg ---
    ("LeftUpperLeg", "Pelvis"),
    ("LeftLowerLeg", "LeftUpperLeg"),
    ("LeftFoot", "LeftLowerLeg"),
    ("LeftToe", "LeftFoot"),
]

bone_orientations_map = {
    "Pelvis": 0,
    "L5": 0,
    "L3": 0,
    "T12": 0,
    "T8": 0,
    "Neck": 0,
    "Head": 0,
    "RightShoulder": 0,
    "RightUpperArm": 0,
    "RightForeArm": 0,
    "RightHand": 0,
    "LeftShoulder": 0,
    "LeftUpperArm": 0,
    "LeftForeArm": 0,
    "LeftHand": 0,
    "RightUpperLeg": 0,
    "RightLowerLeg": 0,
    "RightFoot": 0,
    "RightToe": 0,
    "LeftUpperLeg": 0,
    "LeftLowerLeg": 0,
    "LeftFoot": 0,
    "LeftToe": 0,
    "Prop1": 0,
    "Prop2": 0,
    "Prop3": 0,
    "Prop4": 0,
    # --- Assuming 4 props (24-27), the Carpus of the left hand starts at 28 ---
    "LeftCarpus": 0,
    "LeftFirstMC": 0,
    "LeftFirstPP": 0,
    "LeftFirstDP": 0,
    "LeftSecondMC": 0,
    "LeftSecondPP": 0,
    "LeftSecondMP": 0,
    "LeftSecondDP": 0,
    "LeftThirdMC": 0,
    "LeftThirdPP": 0,
    "LeftThirdMP": 0,
    "LeftThirdDP": 0,
    "LeftFourthMC": 0,
    "LeftFourthPP": 0,
    "LeftFourthMP": 0,
    "LeftFourthDP": 0,
    "LeftFifthMC": 0,
    "LeftFifthPP": 0,
    "LeftFifthMP": 0,
    "LeftFifthDP": 0,
    "RightCarpus": 0,
    "RightFirstMC": 0,
    "RightFirstPP": 0,
    "RightFirstDP": 0,
    "RightSecondMC": 0,
    "RightSecondPP": 0,
    "RightSecondMP": 0,
    "RightSecondDP": 0,
    "RightThirdMC": 0,
    "RightThirdPP": 0,
    "RightThirdMP": 0,
    "RightThirdDP": 0,
    "RightFourthMC": 0,
    "RightFourthPP": 0,
    "RightFourthMP": 0,
    "RightFourthDP": 0,
    "RightFifthMC": 0,
    "RightFifthPP": 0,
    "RightFifthMP": 0,
    "RightFifthDP": 0,
}

# --- Props and Fingers -----------------------------------------------------------------------------------------------------------------
expected_prop_names = [
    "Sword",
    "Crutch",
    "Generic",
    "Golfclub",
    "Gun",
]

possible_prop_segments = [
    "Pelvis",
    "Sternum",
    "Head",
    "RightShoulder",
    "RightUpperArm",
    "RightForeArm",
    "RightHand",
    "LeftShoulder",
    "LeftUpperArm",
    "LeftForeArm",
    "LeftHand",
    "RightUpperLeg",
    "RightLowerLeg",
    "RightFoot",
    "LeftUpperLeg",
    "LeftLowerLeg",
    "LeftFoot",
]

# Offsets from parent joint
finger_joint_offsets = {
    # Metacarpals for fingers are not used in the Xsens model
    "RightCarpus": Vector((0.0, -4.98, 0)),
    "RightFirstMC": Vector((4.28, 4.46, 0.0)),
    "RightFirstPP": Vector((0.71, -2.51, 0.0)),
    "RightFirstDP": Vector((0.0, -2.54, 0.0)),
    "RightSecondMC": Vector((0.0, 0.0, 0.0)),
    "RightSecondPP": Vector((3.62, -4.23, 0.0)),
    "RightSecondMP": Vector((0.09, -2.65, 0.0)),
    "RightSecondDP": Vector((0.07, -1.96, 0.0)),
    "RightThirdMC": Vector((0.0, 0.0, 0.0)),
    "RightThirdPP": Vector((1.30, -4.86, 0.0)),
    "RightThirdMP": Vector((0.0, -2.77, 0.0)),
    "RightThirdDP": Vector((0.0, -2.01, 0.0)),
    "RightFourthMC": Vector((0.0, 0.0, 0.0)),
    "RightFourthPP": Vector((-0.79, -4.62, 0.0)),
    "RightFourthMP": Vector((0.0, -2.30, 0.0)),
    "RightFourthDP": Vector((0.0, -1.92, 0.0)),
    "RightFifthMC": Vector((0.0, 0.0, 0.0)),
    "RightFifthPP": Vector((-2.49, -3.12, 0.0)),
    "RightFifthMP": Vector((0.0, -1.97, 0.0)),
    "RightFifthDP": Vector((0.0, -1.67, 0.0)),
    "LeftCarpus": Vector((0.0, 4.98, 0.0)),
    "LeftFirstMC": Vector((4.28, -4.46, 0.0)),
    "LeftFirstPP": Vector((0.71, 2.51, 0.0)),
    "LeftFirstDP": Vector((0.0, 2.54, 0.0)),
    "LeftSecondMC": Vector((0.0, 0.0, 0.0)),
    "LeftSecondPP": Vector((3.62, 4.23, 0.0)),
    "LeftSecondMP": Vector((0.09, 2.65, 0.0)),
    "LeftSecondDP": Vector((0.07, 1.96, 0.0)),
    "LeftThirdMC": Vector((0.0, 0.0, 0.0)),
    "LeftThirdPP": Vector((1.30, 4.86, 0.0)),
    "LeftThirdMP": Vector((0.0, 2.77, 0.0)),
    "LeftThirdDP": Vector((0.0, 2.01, 0.0)),
    "LeftFourthMC": Vector((0.0, 0.0, 0.0)),
    "LeftFourthPP": Vector((-0.79, 4.62, 0.0)),
    "LeftFourthMP": Vector((0.0, 2.30, 0.0)),
    "LeftFourthDP": Vector((0.0, 1.92, 0.0)),
    "LeftFifthMC": Vector((0.0, 0.0, 0.0)),
    "LeftFifthPP": Vector((-2.49, 3.12, 0.0)),
    "LeftFifthMP": Vector((0.0, 1.97, 0.0)),
    "LeftFifthDP": Vector((0.0, 1.67, 0.0)),
}

# -Remapping Dictionaries ---------------------------------------------------------------------------------------------------------------
common_remapings = {
    1: ["Pelvis", "Hips", "pelvis", "hips"],
    2: ["L5", "Spine0", "Spine", "spine_00"],
    3: ["L3", "Spine1", "spine_01"],
    4: ["T12", "Spine2", "spine_02"],
    5: ["T8", "Spine3", "spine_03"],
    6: ["Neck", "neck", "Neck0", "Neck1", "neck_01"],
    7: ["Head", "head", "head_01"],
    8: ["RightShoulder", "RightClavicle", "clavicle_r", "shoulder.R"],
    9: ["RightUpperArm", "RightUpArm", "RightArm", "upperarm_r", "upper_arm.R"],
    10: ["RightForeArm", "RightLowerArm", "RightLowArm", "lowerarm_r", "forearm.R"],
    11: ["RightHand", "hand_r", "hand.R"],
    12: ["LeftShoulder", "LeftClavicle", "clavicle_l", "shoulder.L"],
    13: ["LeftUpperArm", "LeftUpArm", "LeftArm", "upperarm_l", "upper_arm.L"],
    14: ["LeftForeArm", "LeftLowerArm", "LeftLowArm", "lowerarm_l", "forearm.L"],
    15: ["LeftHand", "hand_l", "hand.L"],
    16: ["RightUpperLeg", "RightUpLeg", "RightLeg", "thigh_r", "thigh.R"],
    17: ["RightLowerLeg", "RightLowLeg", "RightLeg", "calf_r", "shin.R"],
    18: ["RightFoot", "foot_r", "foot.R"],
    19: ["RightToe", "RightToeBase", "ball_r", "toe.R", "toe_r"],
    20: ["LeftUpperLeg", "LeftUpLeg", "LeftLeg", "thigh_l", "thigh.L"],
    21: ["LeftLowerLeg", "LeftLowLeg", "LeftLeg", "calf_l", "shin.L"],
    22: ["LeftFoot", "foot_l", "foot.L"],
    23: ["LeftToe", "LeftToeBase", "ball_l", "toe.L", "toe_l"],
    # --- Props ---
    24: ["Prop1"],
    25: ["Prop2"],
    26: ["Prop3"],
    27: ["Prop4"],
    # --- Fingers ---
    28: ["LeftCarpus"],
    29: ["LeftFirstMC"],
    30: ["LeftFirstPP"],
    31: ["LeftFirstDP"],
    32: ["LeftSecondMC"],
    33: ["LeftSecondPP"],
    34: ["LeftSecondMP"],
    35: ["LeftSecondDP"],
    36: ["LeftThirdMC"],
    37: ["LeftThirdPP"],
    38: ["LeftThirdMP"],
    39: ["LeftThirdDP"],
    40: ["LeftFourthMC"],
    41: ["LeftFourthPP"],
    42: ["LeftFourthMP"],
    43: ["LeftFourthDP"],
    44: ["LeftFifthMC"],
    45: ["LeftFifthPP"],
    46: ["LeftFifthMP"],
    47: ["LeftFifthDP"],
    48: ["RightCarpus"],
    49: ["RightFirstMC"],
    50: ["RightFirstPP"],
    51: ["RightFirstDP"],
    52: ["RightSecondMC"],
    53: ["RightSecondPP"],
    54: ["RightSecondMP"],
    55: ["RightSecondDP"],
    56: ["RightThirdMC"],
    57: ["RightThirdPP"],
    58: ["RightThirdMP"],
    59: ["RightThirdDP"],
    60: ["RightFourthMC"],
    61: ["RightFourthPP"],
    62: ["RightFourthMP"],
    63: ["RightFourthDP"],
    64: ["RightFifthMC"],
    65: ["RightFifthPP"],
    66: ["RightFifthMP"],
    67: ["RightFifthDP"],
}
auto_remappings = {
    1: ["Pelvis", "Hips"],
    2: ["L5", "Spine0", "Spine", "Spine_00", "Spine1", "Spine_01"],
    3: ["L3", "Spine1", "Spine_01", "Spine2", "Spine_02"],
    4: ["T12", "Spine2", "Spine_02", "Spine3", "Spine_03"],
    5: ["T8", "Spine3", "Spine_03", "Spine4", "Spine_04"],
    6: ["Neck", "Neck0", "Neck1", "Neck_01"],
    7: ["Head", "Head_01"],
    8: ["RightShoulder", "RightClavicle", "Clavicle_R", "shoulder.R"],
    9: ["RightUpperArm", "RightUpArm", "RightArm", "Upperarm_R", "upper_arm.R"],
    10: ["RightForeArm", "RightLowerArm", "RightLowArm", "Lowerarm_R", "forearm.R"],
    11: ["RightHand", "Hand_R", "hand.R"],
    12: ["LeftShoulder", "LeftClavicle", "Clavicle_L", "shoulder.L"],
    13: ["LeftUpperArm", "LeftUpArm", "LeftArm", "Upperarm_L", "upper_arm.L"],
    14: ["LeftForeArm", "LeftLowerArm", "LeftLowArm", "Lowerarm_L", "forearm.L"],
    15: ["LeftHand", "Hand_L", "hand.L"],
    16: ["RightUpperLeg", "RightUpLeg", "RightLeg", "Thigh_R", "thigh.R"],
    17: ["RightLowerLeg", "RightLowLeg", "RightLeg", "Calf_R", "shin.R"],
    18: ["RightFoot", "Foot_R", "foot.R"],
    19: ["RightToe", "RightToeBase", "Ball_R", "toe.R", "Toe_R"],
    20: ["LeftUpperLeg", "LeftUpLeg", "LeftLeg", "Thigh_L", "thigh.L"],
    21: ["LeftLowerLeg", "LeftLowLeg", "LeftLeg", "Calf_L", "shin.L"],
    22: ["LeftFoot", "Foot_L", "foot.L"],
    23: ["LeftToe", "LeftToeBase", "Ball_L", "toe.L", "Toe_L"],
    # --- Props ---
    24: ["Prop1"],
    25: ["Prop2"],
    26: ["Prop3"],
    27: ["Prop4"],
    # --- Fingers ---
    28: ["LeftCarpus", "LeftHandThumb1"],
    29: ["LeftFirstMC", "LeftHandThumb2", "thumb_01_l", "L Finger0", "thumb.01.L"],
    30: ["LeftFirstPP", "LeftHandThumb3", "thumb_02_l", "L Finger01", "thumb.02.L"],
    31: ["LeftFirstDP", "LeftHandThumb4", "thumb_03_l", "L Finger02", "thumb.03.L"],
    32: ["LeftSecondMC", "LeftHandIndex1", "index_metacarpal_l"],
    33: ["LeftSecondPP", "LeftHandIndex2", "index_01_l", "L Finger1", "f_index.01.L"],
    34: ["LeftSecondMP", "LeftHandIndex3", "index_02_l", "L Finger11", "f_index.02.L"],
    35: ["LeftSecondDP", "LeftHandIndex4", "index_03_l", "L Finger12", "f_index.03.L"],
    36: ["LeftThirdMC", "LeftHandMiddle1", "middle_metacarpal_l"],
    37: ["LeftThirdPP", "LeftHandMiddle2", "middle_01_l", "L Finger2", "f_middle.01.L"],
    38: ["LeftThirdMP", "LeftHandMiddle3", "middle_02_l", "L Finger21", "f_middle.02.L"],
    39: ["LeftThirdDP", "LeftHandMiddle4", "middle_03_l", "L Finger22", "f_middle.03.L"],
    40: ["LeftFourthMC", "LeftHandRing1", "ring_metacarpal_l"],
    41: ["LeftFourthPP", "LeftHandRing2", "ring_01_l", "L Finger3", "f_ring.01.L"],
    42: ["LeftFourthMP", "LeftHandRing3", "ring_02_l", "L Finger31", "f_ring.02.L"],
    43: ["LeftFourthDP", "LeftHandRing4", "ring_03_l", "L Finger32", "f_ring.03.L"],
    44: ["LeftFifthMC", "LeftHandPinky1", "pinky_metacarpal_l"],
    45: ["LeftFifthPP", "LeftHandPinky2", "pinky_01_l", "L Finger4", "f_pinky.01.L"],
    46: ["LeftFifthMP", "LeftHandPinky3", "pinky_02_l", "L Finger41", "f_pinky.02.L"],
    47: ["LeftFifthDP", "LeftHandPinky4", "pinky_03_l", "L Finger42", "f_pinky.03.L"],
    48: ["RightCarpus", "RightHandThumb1"],
    49: ["RightFirstMC", "RightHandThumb2", "thumb_01_r", "R Finger0", "thumb.01.R"],
    50: ["RightFirstPP", "RightHandThumb3", "thumb_02_r", "R Finger01", "thumb.02.R"],
    51: ["RightFirstDP", "RightHandThumb4", "thumb_03_r", "R Finger02", "thumb.03.R"],
    52: ["RightSecondMC", "RightHandIndex1", "index_metacarpal_r"],
    53: ["RightSecondPP", "RightHandIndex2", "index_01_r", "R Finger1", "f_index.01.R"],
    54: ["RightSecondMP", "RightHandIndex3", "index_02_r", "R Finger11", "f_index.02.R"],
    55: ["RightSecondDP", "RightHandIndex4", "index_03_r", "R Finger12", "f_index.03.R"],
    56: ["RightThirdMC", "RightHandMiddle1", "middle_metacarpal_r"],
    57: ["RightThirdPP", "RightHandMiddle2", "middle_01_r", "R Finger2", "f_middle.01.R"],
    58: ["RightThirdMP", "RightHandMiddle3", "middle_02_r", "R Finger21", "f_middle.02.R"],
    59: ["RightThirdDP", "RightHandMiddle4", "middle_03_r", "R Finger22", "f_middle.03.R"],
    60: ["RightFourthMC", "RightHandRing1", "ring_metacarpal_r"],
    61: ["RightFourthPP", "RightHandRing2", "ring_01_r", "R Finger3", "f_ring.01.R"],
    62: ["RightFourthMP", "RightHandRing3", "ring_02_r", "R Finger31", "f_ring.02.R"],
    63: ["RightFourthDP", "RightHandRing4", "ring_03_r", "R Finger32", "f_ring.03.R"],
    64: ["RightFifthMC", "RightHandPinky1", "pinky_metacarpal_r"],
    65: ["RightFifthPP", "RightHandPinky2", "pinky_01_r", "R Finger4", "f_pinky.01.R"],
    66: ["RightFifthMP", "RightHandPinky3", "pinky_02_r", "R Finger41", "f_pinky.02.R"],
    67: ["RightFifthDP", "RightHandPinky4", "pinky_03_r", "R Finger42", "f_pinky.03.R"],
}
